package org.aksw.commons.util.docker;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

/**
 * Argument vector class. Wraps an list of strings.
 */
public record Argv(List<String> argv) {
    public Argv {
        requireNonEmpty(argv);
        argv = List.copyOf(Objects.requireNonNull(argv));
    }

    private static void requireNonEmpty(Collection<?> list) {
        Objects.requireNonNull(list);
        if (list.isEmpty()) {
            throw new IllegalArgumentException("At least on item expected.");
        }
    }

    public static Argv of(List<String> argv) {
        return new Argv(argv);
    }

    public static Argv of(String... argv) {
        return new Argv(List.of(argv));
    }

    public static Argv ofArgs(String command, String ...args) {
        return ofArgs(command, List.of(args));
    }

    public static Argv ofArgs(String command, List<String> args) {
        List<String> argv = new ArrayList<>(1 + args.size());
        argv.add(command);
        argv.addAll(args);
        return new Argv(argv);
    }

    public boolean isEmpty() {
        return argv.isEmpty();
    }

    /** Return the command, or null if this argv instance is empty. */
    public String command() {
        return argv.isEmpty() ? null : argv.get(0);
    }

    public List<String> args() {
        return argv.subList(1, argv.size());
    }

    public String[] toArray() {
        return argv.toArray(String[]::new);
    }

    public String[] argsToArray() {
        return args().toArray(String[]::new);
    }
}
