package org.aksw.jenax.arq.util.var;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.IntPredicate;
import java.util.regex.Pattern;

import org.aksw.commons.collections.generator.Generator;
import org.apache.jena.riot.system.RiotChars;
import org.apache.jena.sparql.core.Var;
import org.apache.jena.sparql.graph.NodeTransform;

public class VarUtils {
    public static final Pattern VARNAME = Pattern.compile("(\\?|\\$)?(?<varname>\\S*)");

    // https://www.w3.org/TR/sparql11-query/#rVARNAME
    // PN_CHARS_BASE ::= [A-Z] | [a-z] | [#x00C0-#x00D6] | [#x00D8-#x00F6] | [#x00F8-#x02FF] | [#x0370-#x037D] | [#x037F-#x1FFF] | [#x200C-#x200D] | [#x2070-#x218F] | [#x2C00-#x2FEF] | [#x3001-#xD7FF] | [#xF900-#xFDCF] | [#xFDF0-#xFFFD] | [#x10000-#xEFFFF]
    // PN_CHARS_U	  ::=  	PN_CHARS_BASE | '_'
    // VARNAME        ::= ( PN_CHARS_U | [0-9] ) ( PN_CHARS_U | [0-9] | #x00B7 | [#x0300-#x036F] | [#x203F-#x2040] )*

    public static boolean isValidFirstCharForVarName(int ch) {
        return RiotChars.isPNChars_U(ch) ||  RiotChars.range(ch, '0', '9');
    }

    public static boolean isValidNonFirstCharForVarName(int ch) {
        return isValidFirstCharForVarName(ch) || ch == 0x00B7 || RiotChars.range(ch, 0x0300, 0x036F) || RiotChars.range(ch, 0x203F, 0x2040);
    }

// TODO Finish - we may want to track all invalid characters
//    public static boolean isValidVarName(String varName) {
//        // NOTE In TARQL there is a comment: "I've omitted UTF-16 character range #x10000-#xEFFFF."
//        int[] codePoints = varName.codePoints().toArray();
//        if (codePoints.length > 0) {
//            int before = codePoints[0];
//            int after = isValidFirstCharForVarName(before) ? before : '_';
//            sb.appendCodePoint(after);
//        }
//
//        for (int i = 1; i < codePoints.length; ++i) {
//            int before = codePoints[i];
//            int after = isValidNonFirstCharForVarName(before) ? before : '_';
//            sb.appendCodePoint(after);
//        }
//    }

    /**
     * Parse the patterns (?|$)\\S* as variables
     *
     * @return
     */
//	public static Var parseVar(String str) {
//		Matcher m = VARNAME.matcher(str);
//		String s = m.find()
//				? m.group(2)
//				: null;
//
//		Var result = Var.alloc(s);
//		return result;
//	}


    /**
     * This method parses the string generated by Map<Var, Var>.toString() back into the Java object.
     * Variable names must not contain symbols '=' and ','
     *
     * @param str
     * @return
     */
    public static Map<Var, Var> parseVarMap(String str) {
        Map<Var, Var> result = new HashMap<>();

        int l = str.length();

        int start = str.startsWith("{") ? 1 : 0;
        int end = str.endsWith("}") ? l - 1 : l;

        String sub = str.substring(start, end);
        String[] entries = sub.split(",");

        for(String entry : entries) {
            String[] kv = entry.split("=", 2);
            Var k = parseVar(kv[0]);
            Var v = parseVar(kv[1]);
            result.put(k, v);
        }

        return result;
    }

    public static Var parseVar(String str) {
        // Remove leading ? of the varName
        String varName = str.trim();
        char c = varName.charAt(0);
        if(c != '?' && c != '$') {
            throw new RuntimeException("var name must start with '?' or '$'");
        }
        varName = varName.substring(1);

        Var result = Var.alloc(varName);
        return result;
    }


    /**
     * Create a generator which yields fresh variables that is not contained in the array 'vars'.
     * The new var name will have the given prefix
     *
     */
    public static Generator<Var> createVarGen(String prefix, Collection<Var> excludeVars) {
        prefix = prefix == null ? "v" : prefix;

        Generator<Var> result = VarGeneratorBlacklist.create(prefix, excludeVars);
        //      Set<//Var> excludeVarNames = getVarNames(excludeVars);
        //var generator = GenSym.create(prefix);
        //var genVarName = new GeneratorBlacklist(generator, excludeVarNames);

        //var result = new VarGen(genVarName);

        return result;
    }

    /**
     * Returns a list of variable names as strings for a given iterable of Var objects.
     *
     * @param vars
     * @return
     */
    public static List<String> getVarNames(Iterable<Var> vars) {
        List<String> result = new ArrayList<String>();

        for(Var var : vars) {
            result.add(var.getName());
        }

        return result;
    }

    public static List<Var> toList(Collection<String> varNames) {
        List<Var> result = new ArrayList<Var>(varNames.size());
        for(String varName : varNames) {
            Var var = Var.alloc(varName);
            result.add(var);
        }

        return result;
    }

    /** Convert a collection of var names into a linked hash set of Vars */
    public static Set<Var> toSet(Collection<String> varNames) {
        Set<Var> result = new LinkedHashSet<>();
        for(String varName : varNames) {
            Var var = Var.alloc(varName);
            result.add(var);
        }

        return result;
    }


    public static List<String> map(Collection<String> varNames, Map<Var, Var> varMap) {
        List<String> result = new ArrayList<String>(varNames.size());
        for(String varName : varNames) {
            Var sourceVar = Var.alloc(varName);
            Var targetVar = varMap.get(sourceVar);

            if(targetVar == null) {
                targetVar = sourceVar;
            }

            String targetVarName = targetVar.getVarName();
            result.add(targetVarName);
        }

        return result;
    }

    public static Var applyNodeTransform(Var var, NodeTransform nodeTransform) {
        Var result = applyNodeTransform(var, nodeTransform, var);
        return result;
    }

    public static Var applyNodeTransform(Var var, NodeTransform nodeTransform, Var defaultVar) {
        Var tmp = (Var)nodeTransform.apply(var);
        Var result = tmp == null ? defaultVar : tmp;
        return result;
    }

    /**
     * Returns a map that maps *each* variable from vbs to a name that does not appear in vas.
     *
     * @param excludeSymmetry if true, exclude mappings from a var in vbs to itself.
     */
    public static Map<Var, Var> createDistinctVarMap(Collection<Var> vas, Collection<Var> vbs, boolean excludeSymmetry, Generator<Var> generator) {

        // Ensure that the generator does not yield a forbidden variable
        Set<Var> forbidden = new HashSet<>();
        forbidden.addAll(vas);
        forbidden.addAll(vbs);
        generator = VarGeneratorBlacklist.create(generator, forbidden); //vas);

        // Rename all variables that are in common
        Map<Var, Var> result = new HashMap<Var, Var>();

        for(Var oldVar : vbs) {
            Var newVar;
            if (vas.contains(oldVar)) {
                newVar = generator.next();
            } else {
                newVar = oldVar;
            }

            boolean isSame = oldVar.equals(newVar);
            if(!(excludeSymmetry && isSame)) {
                result.put(oldVar, newVar);
            }
        }

        return result;
    }

    public static Map<Var, Var> createJoinVarMap(Collection<Var> sourceVars, Collection<Var> targetVars, List<Var> sourceJoinVars, List<Var> targetJoinVars, Generator<Var> generator) {

        if (sourceJoinVars.size() != targetJoinVars.size()) {
            throw new RuntimeException("Cannot join on different number of columns");
        }

        Map<Var, Var> result = VarUtils.createDistinctVarMap(sourceVars, targetVars, true, generator);

        for (int i = 0; i < sourceJoinVars.size(); ++i) {
            Var sourceJoinVar = sourceJoinVars.get(i);
            Var targetJoinVar = targetJoinVars.get(i);

            // Map targetVar to sourceVar
            result.put(targetJoinVar, sourceJoinVar);
            // rename[targetVar.getName()] = sourceVar;
        }

        return result;
    }

    /**
     * Return a new string that has all characters disallowed in SPARQL variable names replaced with underscore ('_').
     */
    public static String safeVarName(String varName) {
        // NOTE In TARQL there is a comment: "I've omitted UTF-16 character range #x10000-#xEFFFF."
        String result = safeIdentifier(varName, '_',
                VarUtils::isValidFirstCharForVarName, VarUtils::isValidNonFirstCharForVarName);
        return result;
    }

    public static String safeIdentifier(String varName, int replacement, IntPredicate isValidChar) {
        return safeIdentifier(varName, replacement, isValidChar, isValidChar);
    }

    /**
     * Return a new string that has all characters disallowed in SPARQL variable names replaced with underscore ('_').
     */
    // Move to aksw-commons
    public static String safeIdentifier(String varName, int replacement, IntPredicate isValidFirstChar, IntPredicate isValidNonFirstChar) {
        // NOTE In TARQL there is a comment: "I've omitted UTF-16 character range #x10000-#xEFFFF."
        StringBuilder sb = new StringBuilder();
        int[] codePoints = varName.codePoints().toArray();
        if (codePoints.length > 0) {
            int before = codePoints[0];
            int after = isValidFirstChar.test(before) ? before : replacement;
            sb.appendCodePoint(after);
        }

        for (int i = 1; i < codePoints.length; ++i) {
            int before = codePoints[i];
            int after = isValidNonFirstChar.test(before) ? before : replacement;
            sb.appendCodePoint(after);
        }

        String result = sb.toString();
        return result.isEmpty() ? null : result;
    }


    /**
     * Create a variable with a safe version of the given name using {@link #safeVarName(String)}.
     */
    public static Var safeVar(String varName) {
        String safeName = safeVarName(varName);
        return safeName == null ? null : Var.alloc(safeName);
    }
}
