package org.aksw.jena_sparql_api.rx.script;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import io.github.classgraph.ClassGraph;
import io.github.classgraph.Resource;
import io.github.classgraph.ResourceList;
import io.github.classgraph.ScanResult;


// NOTES
// Jena has a class path resource resolver, and I think it should be reused.
// However, it lacks the globbing support.
// So it think the role of ClassGraph is to be preprocessor that is only used for resolving
// globbing of class path resources (at least for the time being).

public class ResourceResolver {

    /** Try to resolve the glob to a list of resources on the class path. */
    public static List<String> resolveClassPathResources(String glob) {
        String normalized = glob.replace('\\', '/');
        String prefix = longestLiteralPrefix(normalized);
        // Remove leading slashes from the prefix. ClassGraph resources don't start with a slash.
        normalized = normalized.replaceAll("^//+", "");

        List<String> result = new ArrayList<>();
        try (ScanResult scanResult = new ClassGraph().acceptPaths(prefix).scan()) {
            ResourceList resourceList = scanResult.getResourcesMatchingWildcard(normalized);
            //ResourceList resourceList = scanResult.getAllResources();
            for (Resource r : resourceList) {
                result.add(r.getPath());
            }
        }
        Collections.sort(result);
        return result;
    }

    public static List<SimpleResource> resolveResources(String pathPattern) throws IOException {
        List<SimpleResource> result = new ArrayList<>();

        // First, attempt to resolve filesystem matches
        PathMatcher matcher = FileSystems.getDefault().getPathMatcher("glob:" + pathPattern);
        try (var files = Files.walk(Paths.get("."))) {
            for (Path path : files.toList()) {
                if (matcher.matches(path) && Files.isRegularFile(path)) {
                    result.add(new SimpleResourceFromPath(path)); // Wrap Path into a Resource-like adapter
                }
            }
        }

        // If no filesystem match, fall back to classpath resources
        String normalized = pathPattern.replace('\\', '/');
        boolean foundFsFiles = !result.isEmpty();
        if (!foundFsFiles) {
            String prefix = longestLiteralPrefix(normalized);
            // Remove leading slashes from the prefix. ClassGraph resources don't start with a slash.
            normalized = normalized.replaceAll("^//+", "");

            try (ScanResult scanResult = new ClassGraph().acceptPaths(prefix).scan()) {
                ResourceList resourceList = scanResult.getResourcesMatchingWildcard(normalized);
                //ResourceList resourceList = scanResult.getAllResources(); // scanResult.getResourcesMatchingWildcard(normalized);
                for (Resource r : resourceList) {
                    result.add(new SimpleResourceFromClassGraphResource(r));
                }
            }
        }

        return result;
    }

    /** Longest literal (meta-free) prefix of a Unix-style glob. */
    public static String longestLiteralPrefix(String glob) {
        if (glob == null) {
            throw new NullPointerException("glob");
        }

        int cut = StringUtils.indexOfAny(glob, '*', '?', '[');
        if (cut == -1) {
            cut = glob.length();
        }

        // Back up over any trailing slashes.
        int end = cut;
        while (end > 0 && glob.charAt(end - 1) == '/') {
            end--;
        }
        String prefix = glob.substring(0, end);
        return prefix;
    }

    public static void main(String[] args) throws IOException {
        List<String> resolved = resolveClassPathResources("void/sportal/compact/*.rq");
        System.out.println(resolved);
//        List<SimpleResource> resources = resolveResources("void/sportal/compact/*.rq");
//
//        for (SimpleResource resource : resources) {
//            System.out.println("Opened: " + resource.getQualifiedName());
////            try (var in = resource.open()) { // safe per-resource handling
////                System.out.println("Opened: " + resource.getQualifiedName());
////            }
//        }
        System.out.println("done.");
    }
}

