package org.aksw.conjure.dataengine;

import java.io.Closeable;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;

import org.aksw.commons.io.util.FileUtils;
import org.aksw.commons.io.util.PathUtils;
import org.aksw.commons.io.util.symlink.SymbolicLinkStrategies;
import org.aksw.commons.util.exception.FinallyRunAll;
import org.aksw.difs.builder.DifsFactory;
import org.aksw.difs.system.domain.StoreDefinition;
import org.aksw.jenax.arq.engine.quad.RDFConnectionFactoryQuadForm;
import org.aksw.jenax.arq.service.vfs.ServiceExecutorFactoryRegistratorVfs;
import org.aksw.jenax.dataaccess.sparql.engine.RDFEngine;
import org.aksw.jenax.dataaccess.sparql.engine.RDFEngines;
import org.aksw.jenax.dataaccess.sparql.factory.dataengine.RDFEngineFactoryLegacyBase;
import org.aksw.jenax.dataaccess.sparql.factory.datasource.RdfDataSourceSpecBasic;
import org.aksw.jenax.dataaccess.sparql.factory.datasource.RdfDataSourceSpecBasicFromMap;
import org.aksw.jenax.dataaccess.sparql.linksource.RDFLinkSource;
import org.aksw.jenax.dataaccess.sparql.linksource.RDFLinkSourceOverDatasetGraph;
import org.apache.jena.query.ARQ;
import org.apache.jena.query.Dataset;
import org.apache.jena.query.DatasetFactory;
import org.apache.jena.rdf.model.ModelFactory;
import org.apache.jena.rdfconnection.RDFConnection;
import org.apache.jena.rdflink.RDFLink;
import org.apache.jena.rdflink.RDFLinkAdapter;
import org.apache.jena.sparql.core.DatasetGraph;
import org.apache.jena.sparql.util.Context;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDFEngineFactoryDifs
    extends RDFEngineFactoryLegacyBase
{
    private static final Logger logger = LoggerFactory.getLogger(RDFEngineFactoryDifs.class);

    @Override
    public RDFEngine create(Map<String, Object> config) throws Exception {

        RdfDataSourceSpecBasic spec = RdfDataSourceSpecBasicFromMap.wrap(config);

        if (spec.getLocation() == null) {
            throw new IllegalArgumentException("Dataset-in-FileSystem (Difs) engine requires the location of a store config file");
        }

        Entry<Path, Closeable> fsInfo = PathUtils.resolveFsAndPath(spec.getLocationContext(), spec.getLocation());
        Path confFile = fsInfo.getKey();

        boolean canWrite = confFile.getFileSystem().equals(FileSystems.getDefault());

        Context cxt = ARQ.getContext().copy();
        ServiceExecutorFactoryRegistratorVfs.register(cxt);


        // If the config does not exist then delete it upon completion

        Path basePath = confFile.getParent();
        if (basePath == null) {
            throw new IllegalArgumentException("Location must be a file");
        }

        // TODO Get default paths from difs factory to ensure consistency
        Path dftStorePath = basePath.resolve("store");
        Path dftIndexPath = basePath.resolve("index");

        boolean deleteWhenDone =
                Boolean.TRUE.equals(spec.isAutoDeleteIfCreated())
                && canWrite
                && !Files.exists(confFile) && !Files.exists(dftStorePath) && !Files.exists(dftIndexPath);

        if (deleteWhenDone) {
            logger.info(String.format("Creating temporary difs store with config file %s (files will be deleted when done)", confFile));
        }
//        else {
//            logger.info(String.format("Connecting to existing difs store using config %s", confFile));
//        }

        Path dftLocksPath = basePath.resolve("locks");
        Path dftTxnsPath = basePath.resolve("txns");

        // Make sure we don't accidently delete possibly unrelated locks / txns folders (better safe than sorry)
        if (deleteWhenDone) {
            for (Path validation : Arrays.asList(dftLocksPath, dftTxnsPath) ) {
                if (Files.exists(validation)) {
                    throw new IllegalStateException("Neither store/index folders nor config file found but either orphaned or unrelated file " + validation + " existed.");
                }
            }
        }

        Path ancestorPath = FileUtils.getFirstExistingAncestor(confFile);
        Files.createDirectories(basePath);

        // Path createDirs = ancestorPath.relativize(basePath);



        StoreDefinition defaultDefinition = ModelFactory.createDefaultModel().createResource().as(StoreDefinition.class)
                .setStorePath("store")
                .setIndexPath("index")
                .setAllowEmptyGraphs(true);

        DatasetGraph dataset = DifsFactory.newInstance()
            .setStoreDefinition(defaultDefinition)
            .setUseJournal(canWrite)
            .setSymbolicLinkStrategy(SymbolicLinkStrategies.FILE)
            .setConfigFile(confFile)
            .setCreateIfNotExists(true)
            .setMaximumNamedGraphCacheSize(10000)
            .connect();

        RDFLinkSource linkSource = new RDFLinkSourceOverDatasetGraph(dataset) {
            @Override
            public RDFLink newLink() {
                // Legacy adapter
                Dataset ds = DatasetFactory.wrap(getDatasetGraph());
                RDFConnection conn = RDFConnectionFactoryQuadForm.connect(ds, cxt);
                RDFLink r = RDFLinkAdapter.adapt(conn);
                return r;
            }
        };

        Closeable closeAction = () -> {
            if (deleteWhenDone) {
                logger.info(String.format("Deleting difs files based at %s", basePath));
                FinallyRunAll.run(
                    () -> FileUtils.deleteRecursivelyIfExists(dftIndexPath),
                    () -> FileUtils.deleteRecursivelyIfExists(dftStorePath),
                    () -> FileUtils.deleteRecursivelyIfExists(dftTxnsPath),
                    () -> FileUtils.deleteRecursivelyIfExists(dftLocksPath),
                    () -> Files.deleteIfExists(confFile),
                    () -> FileUtils.deleteEmptyFolders(basePath, ancestorPath, false),
                    () -> fsInfo.getValue().close()
                );
            }
        };

        RDFEngine result = RDFEngines.of(linkSource, closeAction);
        return result;
    }
}
