/*
 *  BasicPatternElement.java - transducer class
 *
 *  Copyright (c) 1995-2010, The University of Sheffield. See the file
 *  COPYRIGHT.txt in the software or at http://gate.ac.uk/gate/COPYRIGHT.txt
 *
 *  This file is part of GATE (see http://gate.ac.uk/), and is free
 *  software, licenced under the GNU Library General Public License,
 *  Version 2, June 1991 (in the distribution as file licence.html,
 *  and also available at http://gate.ac.uk/gate/licence.html).
 *
 *  Hamish Cunningham, 24/07/98
 *
 *  $Id: BasicPatternElement.java 12006 2009-12-01 17:24:28Z thomas_heitz $
 */


package gate.jape;

import java.util.*;

import gate.*;
import gate.annotation.AnnotationSetImpl;
import gate.util.*;


/**
  * A pattern element within curly braces. Has a set of Constraint,
  * which all must be satisfied at whatever position the element is being
  * matched at.
  */
public class BasicPatternElement
extends PatternElement implements JapeConstants, java.io.Serializable
{
  /** Debug flag */
  private static final boolean DEBUG = false;

  /** A set of Constraint. Used during parsing. */
  private ArrayList<Constraint> constraints1;

  /** A set of Constraint. Used during matching. */
  private Constraint[] constraints2;

  /** A map of constraint annot type to constraint. Used during parsing. */
  private HashMap<Object, Constraint> constraintsMap;

  /** Cache of the last position we failed at (-1 when none). */
  private int lastFailurePoint = -1;

  /** The position of the next available annotation of the type required
    * by the first constraint.
    */
  //private MutableInteger nextAvailable = new MutableInteger();

  /** The set of annotations we have matched. */
  private AnnotationSet matchedAnnots;

  /** Construction. */
  public BasicPatternElement() {
    constraintsMap = new HashMap<Object, Constraint>();
    constraints1 = new ArrayList<Constraint>();
    lastFailurePoint = -1;
    //nextAvailable = new MutableInteger();
    matchedAnnots = new AnnotationSetImpl((Document) null);
  } // construction

  /** Need cloning for processing of macro references. See comments on
    * <CODE>PatternElement.clone()</CODE>
    */
  public Object clone() {
    BasicPatternElement newPE = (BasicPatternElement) super.clone();
    newPE.constraintsMap = (HashMap<Object, Constraint>) constraintsMap.clone();
    newPE.constraints1 = new ArrayList<Constraint>();
    int consLen = constraints1.size();
    for(int i = 0; i < consLen; i++)
      newPE.constraints1.add(
        (Constraint)constraints1.get(i).clone()
      );
//    newPE.matchedAnnots = new AnnotationSetImpl((Document) null);
//    newPE.matchedAnnots.addAll(matchedAnnots);
    return newPE;
  } // clone

  /** Add a constraint. Ensures that only one constraint of any given
    * annotation type and negation state exists.
    */
  public void addConstraint(Constraint newConstraint) {
    /* if a constraint with the same negation state as this constraint is
     * already mapped, put it's attributes on the existing constraint, else
     * add it
     */
    String annotType = newConstraint.getAnnotType();
    Pair typeNegKey = new Pair(annotType, newConstraint.isNegated());

    Constraint existingConstraint = constraintsMap.get(typeNegKey);
    if(existingConstraint == null) {
      constraintsMap.put(typeNegKey, newConstraint);
      constraints1.add(newConstraint);
    }
    else {
      existingConstraint.addAttributes(newConstraint.getAttributeSeq());
    }
  } // addConstraint


  /**
   * Indicates whether this constraint deals with only one type of annotation or
   * multiple types.
   */
  public boolean isMultiType() {
      return constraints2 != null ? constraints2.length > 1 :
             constraints1 != null ? constraints1.size() > 1 :
             false;
  }

  /** Finish: replace dynamic data structures with Java arrays; called
    * after parsing.
    */
  public void finish() {
    int j=0;
    constraints2 = new Constraint[constraints1.size()];
    for(Constraint c : constraints1 ) {
      constraints2[j] = c;
      constraints2[j++].finish();
    }
    constraints1 = null;
  } // finish



  /** Create a string representation of the object. */
  public String toString() {
    StringBuffer result = new StringBuffer("{");
    Constraint[] constraints = getConstraints();
    for(int i = 0; i<constraints.length; i++){
      result.append(constraints[i].shortDesc() + ",");
    }
    result.setCharAt(result.length() -1, '}');
    return result.toString();
  }

  /** Create a string representation of the object. */
  public String toString(String pad) {
    String newline = Strings.getNl();
    String newPad = Strings.addPadding(pad, INDENT_PADDING);

    StringBuffer buf = new StringBuffer(pad +
      "BPE: lastFailurePoint(" + lastFailurePoint + "); constraints("
    );

    // constraints
    if(constraints1 != null) {
      for(int len = constraints1.size(), i = 0; i < len; i++)
        buf.append(
          newline + constraints1.get(i).getDisplayString(newPad)
        );
    } else {
      for(int len = constraints2.length, i = 0; i < len; i++)
        buf.append(newline + constraints2[i].getDisplayString(newPad));
    }

    // matched annots
//    buf.append(
//      newline + pad + "matchedAnnots: " + matchedAnnots +
//      newline + pad + ") BPE."
//    );

    return buf.toString();
  } // toString

  /**
    * Returns a short description.
    */
  public String shortDesc() {
    String res = "";
    if(constraints1 != null) {
      for(int len = constraints1.size(), i = 0; i < len; i++)
        res += constraints1.get(i).toString();
    } else {
      for(int len = constraints2.length, i = 0; i < len; i++)
        res += constraints2[i].shortDesc();
    }
    return res;
  }

  /**
   * Get the current list of unfinished Constraint objects. This
   * can only be used before the finish() method is used.
   * @return the array list of constraint objects. Will be null after
   * the finish() method has been used.
   */
  public ArrayList<Constraint> getUnfinishedConstraints() {
    return constraints1;
  }

  /**
   * Get the finished Constraint objects. Can only be used after the
   * finish() method has been used.
   * @return an array of constraint objects. Will be null before the
   * finish() method has been used.
   */
  public Constraint[] getConstraints(){
    return constraints2;
  }
} // class BasicPatternElement

