/*
 *  TestCreoleAnnotationHandler.java
 *
 *  Copyright (c) 2008, The University of Sheffield.
 *
 *  This file is part of GATE (see http://gate.ac.uk/), and is free
 *  software, licenced under the GNU Library General Public License,
 *  Version 2, June 1991 (in the distribution as file licence.html,
 *  and also available at http://gate.ac.uk/gate/licence.html).
 *
 *  Ian Roberts, 27/Jul/2008
 *
 *  $Id: TestCreoleAnnotationHandler.java 9845 2008-08-25 22:23:24Z ian_roberts $
 */

package gate.creole;

import java.net.URL;
import gate.Gate;
import gate.util.GateException;

import junit.framework.*;
import org.custommonkey.xmlunit.*;
import org.jdom.input.SAXBuilder;
import org.jdom.output.DOMOutputter;
import org.jdom.output.XMLOutputter;
import org.jdom.output.Format;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

/**
 * Test for the CreoleAnnotationHandler, compares the XML produced by the
 * annotation handler to an expected result.
 */
public class TestCreoleAnnotationHandler extends TestCase {

  private SAXBuilder jdomBuilder = new SAXBuilder();

  private DOMOutputter jdom2dom = new DOMOutputter();

  private DocumentBuilderFactory jaxpFactory = DocumentBuilderFactory.newInstance();

  /** Construction */
  public TestCreoleAnnotationHandler(String name) throws GateException {
    super(name);
  }

  public void setUp() throws Exception {
    // Initialise the GATE library and creole register
    Gate.init();

    XMLUnit.setIgnoreComments(true);
    XMLUnit.setIgnoreWhitespace(true);
    XMLUnit.setIgnoreAttributeOrder(true);
  }

  /**
   * Take a skeleton creole.xml file, process the annotations on the classes it
   * mentions and compare the resulting XML to the expected result.
   */
  public void testCreoleAnnotationHandler() throws Exception {
    URL originalUrl = Gate.getUrl("tests/creole-annotation-handler/initial-creole.xml");
    org.jdom.Document creoleXml =
      jdomBuilder.build(originalUrl.openStream());
    
    CreoleAnnotationHandler processor = new CreoleAnnotationHandler(originalUrl);
    processor.processAnnotations(creoleXml);

    URL expectedURL = Gate.getUrl("tests/creole-annotation-handler/expected-creole.xml");

    // XMLUnit requires the expected and actual results as W3C DOM rather than
    // JDOM
    DocumentBuilder docBuilder = jaxpFactory.newDocumentBuilder();
    org.w3c.dom.Document targetXmlDOM =
      docBuilder.parse(expectedURL.openStream());

    org.w3c.dom.Document actualXmlDOM = jdom2dom.output(creoleXml);

    Diff diff = XMLUnit.compareXML(targetXmlDOM, actualXmlDOM);

    // compare parameter elements with the same NAME, resources with the same
    // CLASS, and all other elements that have the same element name
    diff.overrideElementQualifier(new ElementNameQualifier() {
      public boolean qualifyForComparison(Element control, Element test) {
        if("PARAMETER".equals(control.getTagName()) && "PARAMETER".equals(test.getTagName())) {
          return control.getAttribute("NAME").equals(test.getAttribute("NAME"));
        }
        else if("RESOURCE".equals(control.getTagName()) && "RESOURCE".equals(test.getTagName())) {
          String controlClass = findClass(control);
          String testClass = findClass(test);
          return (controlClass == null) ? (testClass == null)
                    : controlClass.equals(testClass);
        }
        else {
          return super.qualifyForComparison(control, test);
        }
      }

      private String findClass(Element resource) {
        Node node = resource.getFirstChild();
        while(node != null && !"CLASS".equals(node.getNodeName())) {
          node = node.getNextSibling();
        }

        if(node != null) {
          return node.getTextContent();
        }
        else {
          return null;
        }
      }
    });

    // do the comparison!
    boolean match = diff.similar();
    if(!match) {
      // if comparison failed, output the "actual" result document for
      // debugging purposes
      new XMLOutputter(Format.getPrettyFormat()).output(creoleXml, System.err);
    }

    assertTrue("XML produced by annotation handler does not match expected: "
        + diff, match);
  }

  /** Test suite routine for the test runner */
  public static Test suite() {
    return new TestSuite(TestCreoleAnnotationHandler.class);
  } // suite

}
