/**
 * EasyBeans
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XMLUtils.java 6181 2012-03-05 12:36:30Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.xml;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Class with some useful methods on XML document.
 */
public final class XMLUtils {

    /**
     * Utility class, no constructor.
     */
    private XMLUtils() {
    }

    /**
     * Returns the value of the given node. (value trimed by default)
     * @param base the element from where to search.
     * @param name of the element to get.
     * @return the value of this element.
     */
    public static String getStringValueElement(final Element base, final String name) {
        return getStringValueElement(base, name, true);
    }

    /**
     * Returns the value of the given node.
     * @param base the element from where to search.
     * @param name of the element to get.
     * @param trim trim or not the value
     * @return the value of this element.
     */
    public static String getStringValueElement(final Element base, final String name, boolean trim) {
        String value = null;

        // Get element
        NodeList list = base.getElementsByTagName(name);
        if (list.getLength() == 1) {
            Element element = (Element) list.item(0);
            Node node = element.getFirstChild();
            if (node != null) {
                value = node.getNodeValue();
            }
        } else if (list.getLength() > 1) {
            throw new IllegalStateException("Element '" + name + "' on '" + base + "' should be unique but there are '"
                    + list.getLength() + "' elements");
        }

        if ((value != null) && trim) {
            value = value.trim();
        }
        return value;
    }

    /**
     * Returns the value of the attribute of the given element.
     * @param base the element from where to search.
     * @param name of the attribute to get.
     * @return the value of this element.
     */
    public static String getAttributeValue(final Element base, final String name) {

        // get attribute of this element...
        NamedNodeMap mapAttributes = base.getAttributes();
        Node node = mapAttributes.getNamedItem(name);
        if (node != null) {
            return node.getNodeValue();
        }
        return null;
    }

    /**
     * Returns the value of the given node or <code>null</code> if the node is not found.
     * @param ns the namespace.
     * @param base the element from where to search.
     * @param name of the element to get.
     * @return the value of this element or null if the element does not exists.
     */
    public static String getStringValueElement(final String ns, final Element base, final String name) {
        return getStringValueElement(ns, base, name, true);
    }

    /**
     * Returns the value of the given node or <code>null</code> if the node is not found.
     * @param ns the namespace.
     * @param base the element from where to search.
     * @param name of the element to get.
     * @param trim trim or not the returned value
     * @return the value of this element or null if the element does not exists.
     */
    public static String getStringValueElement(final String ns, final Element base, final String name, final boolean trim) {
        String value = null;

        // Get element
        NodeList list = base.getElementsByTagNameNS(ns, name);
        if (list.getLength() == 1) {
            Element element = (Element) list.item(0);
            Node node = element.getFirstChild();
            if (node != null) {
                value = node.getNodeValue();
            }
        } else if (list.getLength() > 1) {
            throw new IllegalStateException("Element '" + name + "' on '" + base + "' should be unique but there are '"
                    + list.getLength() + "' elements");
        }

        if ((value != null) && trim) {
            value = value.trim();
        }

        return value;
    }

    /**
     * Returns the value of the child node with the given name.
     * @param base the element from where to search.
     * @param name of the element to get.
     * @return the value of this element.
     */
    public static String getChildStringValueForElement(final Element base, final String name) {
        String value = null;
        NodeList nodeList = base.getChildNodes();
        if (nodeList.getLength() > 0) {
            int length = nodeList.getLength();
            for (int i = 0; i < length; i++) {
                Node node = nodeList.item(i);

                // Get an element, create an instance of the element
                if (Node.ELEMENT_NODE == node.getNodeType()) {
                    if (name.equals(node.getNodeName())) {
                        // Get value of this child
                        Node elNode = ((Element) node).getFirstChild();
                        if (elNode != null) {
                            value = elNode.getNodeValue();
                            break;
                        }
                    }
                }
            }
        }
        return value;
    }


    /**
     * Returns a Properties object matching the given node.
     * Parse the following XML structure:<br/>
     * <code>
     *   [base element]
     *     &lt;ns:name> (only 1)
     *       &lt;ns:property @name @value /> (0..n)
     * </code>
     * @param ns the namespace.
     * @param base the element from where to search.
     * @param name of the element to get.
     * @return the value of this element.
     */
    public static Properties getPropertiesValueElement(final String ns, final Element base, final String name) {
        Properties returnedProperties = new Properties();

        // Get element
        NodeList list = base.getElementsByTagNameNS(ns, name);
        if (list.getLength() == 1) {
            Element element = (Element) list.item(0);

            // Get property element
            NodeList properties = element.getElementsByTagNameNS(ns, "property");

            // If properties is present, analyze them and add them
            if (properties.getLength() > 0) {
                for (int i = 0; i < properties.getLength(); i++) {
                    Element elemProperty = (Element) properties.item(i);
                    String pName = getAttributeValue(elemProperty, "name");
                    String pValue = getAttributeValue(elemProperty, "value");
                    if (pName != null && pValue != null) {
                        returnedProperties.setProperty(pName, pValue);
                    }

                }
            }
        } else if (list.getLength() > 1) {
            throw new IllegalStateException("Element '" + name + "' on '" + base + "' should be unique but there are '"
                    + list.getLength() + "' elements");
        }

        return returnedProperties;
    }

    /**
     * Returns a list of value for the given node.
     * @param ns the namespace.
     * @param base the element from where to search.
     * @param name of the element to get.
     * @return the list of value of this element.
     */
    public static List<String> getStringListValueElement(final String ns, final Element base, final String name) {
        List<String> returnedlist = new ArrayList<String>();

        // Get element
        NodeList list = base.getElementsByTagNameNS(ns, name);
        int length = list.getLength();

        // Get all values of all elements
        if (length > 0) {
            for (int i = 0; i < length; i++) {
                Element element = (Element) list.item(i);
                Node node = element.getFirstChild();
                if (node != null) {
                    String value = node.getNodeValue();
                    if (value != null) {
                        value = value.trim();
                    }
                    returnedlist.add(value);
                }
            }
        }
        return returnedlist;
    }

    /**
     * Returns a list of value for the given node.
     * @param base the element from where to search.
     * @param name of the element to get.
     * @return the list of value of this element.
     */
    public static List<String> getStringListValueElement(final Element base, final String name) {
        List<String> returnedlist = new ArrayList<String>();

        // Get element
        NodeList list = base.getElementsByTagName(name);
        int length = list.getLength();

        // Get all values of all elements
        if (length > 0) {
            for (int i = 0; i < length; i++) {
                Element element = (Element) list.item(i);
                Node node = element.getFirstChild();
                if (node != null) {
                    String value = node.getNodeValue();
                    if (value != null) {
                        value = value.trim();
                    }
                    returnedlist.add(value);
                }
            }
        }
        return returnedlist;
    }

    /**
     * Extract a node's child value as a QName.<br/>
     * <code>
     *   [base element (declaring the prefix/namespace mapping)]
     *     #text-value: "prefix:local-part"
     * </code>
     * @param element the element from where to search.
     * @return The node's value as a QName.
     */
    public static QName getChildValueAsQName(final Element base, final String name) {

        QName value = null;
        NodeList nodeList = base.getChildNodes();
        if (nodeList.getLength() > 0) {
            int length = nodeList.getLength();
            for (int i = 0; i < length; i++) {
                Node node = nodeList.item(i);

                // Get an element, create an instance of the element
                if (Node.ELEMENT_NODE == node.getNodeType()) {
                    if (name.equals(node.getNodeName())) {
                        // Get value of this child
                        value = getValueAsQName((Element) node);
                    }
                }
            }
        }
        return value;
    }

    /**
     * Extract a node's value as a QName.<br/>
     * <code>
     *   [base element (declaring the prefix/namespace mapping)]
     *     #text-value: "prefix:local-part"
     * </code>
     * @param base the element from where to search.
     * @return The node's value as a QName.
     */
    public static QName getValueAsQName(final Element base) {

        // Extract the node's value
        Node node = base.getFirstChild();
        String text = null;
        if (node != null) {
            text = node.getNodeValue();
        }

        // Check that there the value of the element is not null
        if (text == null) {
            throw new IllegalArgumentException("No QName found in the body of the tag " + base.getLocalName());
        }

        // Extract namespace and localpart
        int colonIndex = text.lastIndexOf(":");
        if (colonIndex == -1) {
            // The QName is only formed of a local-part
            return new QName(text.trim());
        }

        String namespacePrefix = text.substring(0, colonIndex);
        String localPart = text.substring(colonIndex + 1, text.length());

        // Lookup the namespace
        String namespaceURI = base.lookupNamespaceURI(namespacePrefix);
        if (namespaceURI == null) {
            throw new IllegalArgumentException("NamespaceURI not found from prefix '" + namespacePrefix
                                               + "' for element " + base.getLocalName() + " node-value: '" + text + "'");
        }

        return new QName(namespaceURI, localPart, namespacePrefix);

    }

    /**
     * Creates a new empty Document.
     * @param namespaceAware <code>true</code> if the created Document have to support namespaces.
     * @return a new empty Document
     */
    public static Document newDocument(final boolean namespaceAware) {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(namespaceAware);
        factory.setValidating(false);
        DocumentBuilder builder = null;
        Document doc = null;
        try {
            builder = factory.newDocumentBuilder();
            doc = builder.newDocument();
        } catch (ParserConfigurationException e) {
            throw new IllegalStateException("Cannot create an empty document."
                    + " Note that this error should never happen!", e);
        }

        return doc;
    }

}
