/**
 * EasyBeans
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SchemaEntityResolver.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.xml;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Entity resolver allowing to find schema within the classloader.
 * @author Florent Benoit
 */
public class SchemaEntityResolver implements EntityResolver {

    /**
     * Map where the schemas URLs are stored.
     */
    private Map<String, String> schemasUrls = null;

    /**
     * Map where the DTDs URLs are stored.
     */
    private Map<String, String> dtdsUrls = null;

    /**
     * Constructor. Finds the XSD with classloader.
     * @param schemas the name of the schemas to resolve.
     */
    public SchemaEntityResolver(final String[] schemas) {
        this(schemas, null, null);
    }

    /**
     * Constructor. Finds the XSD with classloader.
     * @param schemas the name of the schemas to resolve.
     * @param dtdPublicIds the PublicId
     * @param dtdsFileNames the name of the DTDs
     */
    public SchemaEntityResolver(final String[] schemas, final String[] dtdPublicIds, final String[] dtdsFileNames) {
        this.schemasUrls = new HashMap<String, String>();
        this.dtdsUrls = new HashMap<String, String>();
        URL url = null;
        for (int i = 0; i < schemas.length; i++) {
            url = getURL(schemas[i]);
            String urlString = url.toString();
            String id = urlString.substring(urlString.lastIndexOf('/') + 1);
            schemasUrls.put(id, urlString);
        }
        // add DTDs (if any)
        addDtds(dtdPublicIds, dtdsFileNames);
    }

    /**
     * Add the mapping between a public Id and the local path of the DTD.
     * @param dtdPublicIds the PublicId
     * @param dtdsFileNames the name of the DTDs
     */
    protected void addDtds(final String[] dtdPublicIds, final String[] dtdsFileNames) {
        if (dtdPublicIds != null && dtdsFileNames != null && (dtdPublicIds.length == dtdsFileNames.length)) {
            for (int i = 0; i < dtdPublicIds.length; i++) {
                String publicId = dtdPublicIds[i];
                String localURL = getURL(dtdsFileNames[i]).toString();
                dtdsUrls.put(publicId, localURL);
            }
        }
    }

    /**
     * Gets an URL for a given resource.
     * @param resourceName the name of the resource
     * @return the requested URL
     */
    protected URL getURL(final String resourceName) {
        URL url = this.getClass().getResource("/" + resourceName);
        if (url == null) {
            throw new IllegalStateException("'" + resourceName + "' was not found in the current classloader !");
        }
        return url;
    }

    /**
     * The Parser will call this method before opening any external entity
     * except the top-level document entity.
     * @param publicId The public identifier of the external entity being
     *        referenced, or null if none was supplied.
     * @param systemId The system identifier of the external entity being
     *        referenced.
     * @return An InputSource object describing the new input source, or null to
     *         request that the parser open a regular URI connection to the
     *         system identifier.
     * @throws SAXException Any SAX exception, possibly wrapping another
     *         exception.
     * @throws IOException A Java-specific IO exception, possibly the result of
     *         creating a new InputStream or Reader for the InputSource.
     */
    public InputSource resolveEntity(final String publicId, final String systemId) throws IOException, SAXException {

        String localPath = null;

        // Don't handle the DTD (we are a schema resolver)
        if (publicId != null) {
            localPath = dtdsUrls.get(publicId);
        } else if (systemId != null) {
            // Can be a schema
            if (systemId.toLowerCase().endsWith(".xsd")) {
                // Retrieve basename
                String baseName = systemId.substring(systemId.lastIndexOf('/') + 1);

                // Registred ?
                localPath = schemasUrls.get(baseName);
            }
        }

        // schema not found
        if (localPath == null) {
            if (publicId != null) {
                throw new SAXException("No DTD found for '" + publicId + "'.");
            }
            throw new SAXException("No XSD found for '" + systemId + "'.");
        }

        // Return the local path source
        return (new InputSource(localPath));
    }

}
