/**
 * EasyBeans
 * Copyright (C) 2006-2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DocumentParser.java 5495 2010-05-17 14:35:20Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.xml;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Collection;
import java.util.Collections;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Allows to parse an xml file.
 * @author Florent Benoit
 */
public final class DocumentParser {

    /**
     * Utility class.
     */
    private DocumentParser() {

    }

    /**
     * Builds a new Document for a given xml file.
     * @param file the XML file.
     * @param isValidating validate or not the xml file ?
     * @param entityResolver the entityResolver used to validate document (if
     *        validating = true)
     * @throws DocumentParserException if creating of builder fails or parsing
     *         fails.
     * @return an application object.
     */
    public static Document getDocument(final File file,
                                       final boolean isValidating,
                                       final EntityResolver entityResolver)
            throws DocumentParserException {

        // Check if the file exists
        if (!file.exists()) {
            throw new DocumentParserException("The file '" + file + "' does not exist");
        }
        try {
            return getDocument(file.toURI().toURL(), isValidating, entityResolver);
        } catch (MalformedURLException e) {
            throw new DocumentParserException("Cannot build an URL for the file '" + file + "'", e);
        }
    }

    /**
     * Builds a new Document for a given xml file.
     * @param url the URL of the XML file.
     * @param isValidating validate or not the xml file ?
     * @param entityResolver the entityResolver used to validate document (if
     *        validating = true)
     * @throws DocumentParserException if creating of builder fails or parsing
     *         fails.
     * @return an application object.
     */
    public static Document getDocument(final URL url,
                                       final boolean isValidating,
                                       final EntityResolver entityResolver)
            throws DocumentParserException {
        // Open connection
        URLConnection urlConnection = null;
        try {
            urlConnection = url.openConnection();
        } catch (IOException e) {
            throw new DocumentParserException("Cannot open a connection on URL '" + url + "'", e);
        }
        urlConnection.setDefaultUseCaches(false);

        // Extract document from the connection's content
        try {
            return getDocument(urlConnection.getInputStream(),
                               isValidating,
                               entityResolver);
        } catch (IOException e) {
            throw new DocumentParserException("Cannot build an input stream reader on URL '" + url + "'", e);
        }
    }


    /**
     * Builds a new Document for a given xml file.
     * @param stream the {@link InputStream} of the XML file.
     * @param isValidating validate or not the xml file ?
     * @param entityResolver the entityResolver used to validate document (if
     *        validating = true)
     * @throws DocumentParserException if creating of builder fails or parsing
     *         fails.
     * @return an application object.
     */
    public static Document getDocument(final InputStream stream,
                                       final boolean isValidating,
                                       final EntityResolver entityResolver)
            throws DocumentParserException {

        // Wraps the InputStream in an InputSource
        Reader reader = new InputStreamReader(stream);
        InputSource inputSource = new InputSource(reader);

        try {
            return getDocument(inputSource, isValidating, entityResolver);
        } finally {
            // close InputStreamReader when parsing is finished
            try {
                reader.close();
            } catch (IOException e) {
                throw new DocumentParserException("Cannot close the Reader of the XML.", e);
            }
        }
    }

    /**
     * Builds a new Document for a given xml file.
     * @param inputSource the {@link InputSource} of the XML file.
     * @param isValidating validate or not the xml file ?
     * @param entityResolver the entityResolver used to validate document (if
     *        validating = true)
     * @throws DocumentParserException if creating of builder fails or parsing
     *         fails.
     * @return an application object.
     */
    public static Document getDocument(final InputSource inputSource,
                                       final boolean isValidating,
                                       final EntityResolver entityResolver)
            throws DocumentParserException {

        // For compatibility with older version
        // TODO Replace this with an empty Collection or null !!!
        Collection<String> schemaLocations = null;
        schemaLocations = Collections.singletonList("http://java.sun.com/xml/ns/persistence "
                                                  + "http://java.sun.com/xml/ns/persistence/persistence_1_0.xsd");
        return getDocument(inputSource, isValidating, entityResolver, schemaLocations);
    }

    /**
     * Builds a new Document for a given xml file.
     * @param inputSource the {@link InputSource} of the XML file.
     * @param isValidating validate or not the xml file ?
     * @param entityResolver the entityResolver used to validate document (if
     *        validating = true)
     * @param schemaLocations the list of external schemaLocation to force (if
     *        validating = true)
     * @throws DocumentParserException if creating of builder fails or parsing
     *         fails.
     * @return an application object.
     */
    public static Document getDocument(final InputSource inputSource,
                                       final boolean isValidating,
                                       final EntityResolver entityResolver,
                                       final Collection<String> schemaLocations)
            throws DocumentParserException {

        // Build factory
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

        // XML files use schemas.
        factory.setNamespaceAware(true);
        factory.setValidating(isValidating);

        // ignore white space can only be set if parser is validating
        if (isValidating) {
            factory.setIgnoringElementContentWhitespace(true);
            factory.setAttribute("http://apache.org/xml/features/validation/schema", Boolean.valueOf(isValidating));
            factory.setAttribute("http://apache.org/xml/features/validation/schema-full-checking", Boolean.valueOf(true));
        }

        // Add schema location
        if (isValidating && (schemaLocations != null)) {

            // Construct schemaLocation value
            StringBuilder sb = new StringBuilder();
            for (String location : schemaLocations) {
                sb.append(location);
                sb.append(" ");
            }
            String value = sb.toString();

            if (!"".equals(value.trim())) {
                factory.setAttribute("http://apache.org/xml/properties/schema/external-schemaLocation", value.trim());
            }
        }

        // Build a document builder
        DocumentBuilder builder = null;
        try {
            builder = factory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw new DocumentParserException("Cannot build a document builder", e);
        }

        // Error handler (throwing exceptions)
        builder.setErrorHandler(new DefaultErrorHandler());

        // Dummy entity resolver if there is none
        if (entityResolver == null) {
            builder.setEntityResolver(new EmptyEntityResolver());
        } else {
            builder.setEntityResolver(entityResolver);
        }

        // Parse the InputSource as a Document
        Document document = null;
        try {
            document = builder.parse(inputSource);
        } catch (SAXException e) {
            throw new DocumentParserException("Cannot parse the XML InputSource.", e);
        } catch (IOException e) {
            throw new DocumentParserException("Cannot parse the XML InputSource.", e);
        }

        return document;

    }
}
