/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PrefixingResolver.java 5461 2010-03-26 16:01:19Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.substitution.resolver;

import java.util.Collections;
import java.util.List;

import org.ow2.util.substitution.IPropertyResolver;
import org.ow2.util.substitution.engine.util.StringUtils;


/**
 * This resolver allows automatic prefixing management of expression.
 * <code>
 *  Properties properties = new Properties();
 *  properties.setProperty("hello", "world");
 *  PropertiesResolver delegate = new PropertiesResolver(properties);
 *
 *  PrefixingResolver resolver = new PrefixingResolver(delegate, "prefix");
 *  resolver.resolve("hello"); // Return null
 *  resolver.resolve("prefix.hello"); // Return "world"
 * </code>
 * @author Guillaume Sauthier
 */
public class PrefixingResolver extends ForwardingResolver {

    /**
     * Delegating resolver.
     */
    private IPropertyResolver delegate;

    /**
     * Prefix to be removed from the expression.
     */
    private List<String> prefixes;

    /**
     * Is this resolver lax ?
     */
    private boolean lax = false;

    public PrefixingResolver(final IPropertyResolver delegate,
                             final String prefix) {
        this(delegate, Collections.singletonList(prefix));
    }

    public PrefixingResolver(final IPropertyResolver delegate,
                             final List<String> prefixes) {
        this.delegate = delegate;
        this.prefixes = prefixes;
    }

    public boolean isLax() {
        return lax;
    }

    public void setLax(boolean lax) {
        this.lax = lax;
    }

    @Override
    protected IPropertyResolver delegate() {
        return delegate;
    }

    /**
     * @param expression a property name
     * @return the value related to the expression, or null if not found.
     */
    public String resolve(String expression) {

        // For each registered prefix, if the expression matches the prefix, try to
        // resolve the expression without the prefix
        for (String prefix : prefixes) {
            if (expression.startsWith(prefix)) {
                String resolved = delegate().resolve(expression.substring(prefix.length()));
                if (resolved != null) {
                    // if valued, stop
                    return resolved;
                }
                // else continue
            }
        }

        // At this point, the expression did not match any prefix or the
        // resolver could not provided values
        if (isLax()) {
            // Delegate the expression as is ...
            return super.resolve(expression);
        }

        return null;
    }
}
