/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CompositeElement.java 5461 2010-03-26 16:01:19Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.substitution.engine.element;

import java.util.ArrayList;
import java.util.List;

import org.ow2.util.substitution.IPropertyResolver;
import org.ow2.util.substitution.engine.IResolvableElement;

/**
 * A CompositeElement is a resolvable element that contains inner resolvable elements.
 * Example:
 * <code>
 *  composite.${value}
 * </code>
 * @author Guillaume Sauthier
 */
public class CompositeElement implements IResolvableElement {

    /**
     * Inner elements.
     */
    private List<IResolvableElement> elements;

    public CompositeElement() {
        elements = new ArrayList<IResolvableElement>();
    }

    public List<IResolvableElement> getElements() {
        return elements;
    }

    /**
     * Resolve the value of this element using the given resolver.
     * The composite simply concatenate the resolved value of all
     * inner resolvable elements
     * @param resolver IPropertyResolver to be used
     * @return a String representation of this resolved element
     */
    public String getValue(final IPropertyResolver resolver) {
        StringBuilder builder = new StringBuilder();

        for (IResolvableElement element : elements) {
            String resolved = element.getValue(resolver);
            if (resolved == null) {
                throw new IllegalArgumentException("Cannot resolve: " + element);
            }
            builder.append(resolved);
        }
        
        return builder.toString();
    }

    @Override
    public String toString() {
        return toString(0);
    }

    /**
     * Produces an indented String representation of this element.
     * @param indent indentation value
     * @return a String representation
     */
    public String toString(final int indent) {
        StringBuilder sb = new StringBuilder();
        for (IResolvableElement element : elements) {
            sb.append(element.toString(indent + 1));
            sb.append("\n");
        }
        return sb.toString();
    }

}
