/**
 * OW2 Util
 * Copyright (C) 2008 SERLI
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeploymentHelper.java 5078 2009-07-30 09:28:43Z fornacif $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.plan.bindings.deploymentplan;

import org.ow2.util.plan.bindings.exceptions.InvalidDeploymentException;

/**
 * Helper class to access properties of the deployment binding class that
 * have element and attribute members.
 * @author Mickaël LEDUQUE
 */
public class DeploymentHelper {

    /**
     * Accesses the Id of the deployment.
     * @param deployment the deployment.
     * @return the Id.
     * @throws InvalidDeploymentException if both the element and attribute are set.
     */
    public static String getId(final Deployment deployment) throws InvalidDeploymentException {
        String id = deployment.getId();
        String idAttr = deployment.getIdAttr();
        return (String) DeploymentHelper.conciliate(id, idAttr, "id");
    }

    /**
     * Accesses the reloadable flag of the deployment.
     * @param deployment the deployment.
     * @return the reloadable value.
     * @throws InvalidDeploymentException
     */
    public static Boolean isReloadable(final Deployment deployment) throws InvalidDeploymentException {
        Boolean reloadable = deployment.isReloadable();
        Boolean reloadableAttr = deployment.isReloadableAttr();
        return (Boolean) DeploymentHelper.conciliate(reloadable, reloadableAttr, "reloadable");
    }

    /**
     * Accesses the repository reference of the deployment.
     * @param deployment the deployment.
     * @return the repository reference.
     * @throws InvalidDeploymentException if both the element and attribute are set.
     */
    public static String getRepositoryRef(final Deployment deployment) throws InvalidDeploymentException {
        String repositoryRef = deployment.getRepositoryRef();
        String repositoryRefAttr = deployment.getRepositoryRefAttr();
        return (String) DeploymentHelper.conciliate(repositoryRef, repositoryRefAttr, "repository-ref");
    }

    /**
     * Accesses the start flag of the deployment.
     * @param deployment the deployment.
     * @return the start value.
     * @throws InvalidDeploymentException if both the element and attribute are set.
     */
    public static Boolean isStart(final Deployment deployment) throws InvalidDeploymentException {
        Boolean start = deployment.isStart();
        Boolean startAttr = deployment.isStartAttr();
        return (Boolean) DeploymentHelper.conciliate(start, startAttr, "start");
    }

    /**
     * Accesses the reference flag of the deployment.
     * @param deployment the deployment.
     * @return the reference value.
     * @throws InvalidDeploymentException if both the element and attribute are set.
     */
    public static Boolean isReference(final Deployment deployment) throws InvalidDeploymentException {
        Boolean reference = deployment.isReference();
        Boolean referenceAttr = deployment.isReferenceAttr();
        return (Boolean) DeploymentHelper.conciliate(reference, referenceAttr, "reference");
    }

    /**
     * Accesses the startlevel flag of the deployment.
     * @param deployment the deployment.
     * @return the startlevel value.
     * @throws InvalidDeploymentException if both the element and attribute are set.
     */
    public static Integer getStartLevel(final Deployment deployment) throws InvalidDeploymentException {
        Integer startLevel = deployment.getStartlevel();
        Integer startLevelAttr = deployment.getStartlevelAttr();
        return (Integer) DeploymentHelper.conciliate(startLevel, startLevelAttr, "startlevel");
    }

    /**
     * Accesses the starttransient flag of the deployment.
     * @param deployment the deployment.
     * @return the starttransient value.
     * @throws InvalidDeploymentException if both the element and attribute are set.
     */
    public static Boolean isStartTransient(final Deployment deployment) throws InvalidDeploymentException {
        Boolean startTransient = deployment.isStarttransient();
        Boolean startTransientAttr = deployment.isStarttransientAttr();
        return (Boolean) DeploymentHelper.conciliate(startTransient, startTransientAttr, "starttransient");
    }

    private static Object conciliate(final Object first, final Object second, final String element) throws InvalidDeploymentException {
        if (first == null) {
            return second;
        } else {
            if (second == null) {
                return first;
            } else {
                throw new InvalidDeploymentException("deployment " + element + " defined by both element and attribute");
            }
        }
    }
}
