/**
 * EasyBeans
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:I18n.java 1465 2007-06-07 16:38:37Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.i18n;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * This class defines the way for getting message with ResourceBundle for
 * different language. Inspired from
 * http://java.sun.com/docs/books/tutorial/i18n/
 * @author Florent Benoit
 */
public final class I18n {

    /**
     * Name of the resource bundle to use.
     */
    private static final String RESOURCE_BUNDLE_NAME = "I18n";

    /**
     * Name of the default package (when a class don't have a package). This is
     * the case for the generated classes.
     */
    private static final String NO_PACKAGE = "NO_PACKAGE";

    /**
     * List of our I18n objects.
     */
    private static Map<String, I18n> bundles = null;

    /**
     * Internal Resource Bundle.
     */
    private ResourceBundle resourceBundle = null;

    /**
     * Constructor (private access, use getInstance instead).
     * @param packageName name of the package for the bundle.
     * @param cl the classloader used to load bundle.
     */
    private I18n(final String packageName, final ClassLoader cl) {

        String bundleName = packageName + "." + RESOURCE_BUNDLE_NAME;

        // Resource bundle is on the form packagename.I18n.properties
        try {
            Locale locale = Locale.getDefault();
            resourceBundle = ResourceBundle.getBundle(bundleName, locale, cl);
        } catch (MissingResourceException mre) {
            resourceBundle = null;
        }
    }

    /**
     * Gets the instance for a given class.
     * @param c the class for which we want a bundle.
     * @return I18n object.
     */
    public static I18n getInstance(final Class<?> c) {
        return getInstance(c, c.getClassLoader());
    }

    /**
     * Gets the instance for a given class.
     * @param c the class for which we want a bundle.
     * @param cl the classloader used to load bundle.
     * @return I18n object.
     */
    public static I18n getInstance(final Class<?> c, final ClassLoader cl) {
        Package p = c.getPackage();
        if (p != null) {
            return getInstance(p.getName(), cl);
        }
        return getInstance(NO_PACKAGE, cl);
    }

    /**
     * Gets the instance for a given package.
     * @param packageName the package for which we want a bundle.
     * @return I18n object.
     */
    public static I18n getInstance(final String packageName) {
        return getInstance(packageName, Thread.currentThread().getContextClassLoader());
    }

    /**
     * Gets the instance for a given package.
     * @param packageName the package for which we want a bundle.
     * @param cl the classloader used to load bundle.
     * @return I18n object.
     */
    public static I18n getInstance(final String packageName, final ClassLoader cl) {
        if (bundles == null) {
            bundles = new HashMap<String, I18n>();
        }

        I18n i18n = bundles.get(packageName);
        if (i18n != null) {
            return i18n;
        }

        i18n = new I18n(packageName, cl);
        bundles.put(packageName, i18n);
        return i18n;
    }

    /**
     * Gets the formatted string with the given arguments.
     * @param key the keystring on which to apply arguments.
     * @param objs list of argument for the formatter.
     * @return the formatted string.
     */
    public String getMessage(final String key, final Object... objs) {
        String pattern = key;
        // No bundle, return key
        if (resourceBundle != null) {
            try {
                pattern = resourceBundle.getString(key);
            } catch (MissingResourceException mre) {
                // key not found, use this pattern.
                pattern = key;
            }
        }
        try {
            return MessageFormat.format(pattern, objs);
        } catch (IllegalArgumentException iae) {
            return key;
        }
    }

}

