/**
 * OW2 Util
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: UnpackDeployableHelper.java 5981 2011-09-28 19:07:40Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.helper;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.archive.api.IFileArchive;
import org.ow2.util.archive.impl.ArchiveManager;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployable.UnknownDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployment.ear.IEARInfo;
import org.ow2.util.ee.deploy.api.deployment.ear.Module;
import org.ow2.util.ee.deploy.api.helper.DeployableHelperException;
import org.ow2.util.ee.deploy.api.helper.IDeployableHelper;
import org.ow2.util.ee.deploy.impl.deployable.EARDeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.UnknownDeployableImpl;
import org.ow2.util.file.FileUtils;
import org.ow2.util.file.FileUtilsException;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.url.URLUtils;

/**
 * Allow to unpack a deployable and build a deployable for this type.
 * @author Florent BENOIT
 */
public final class UnpackDeployableHelper {

    /**
     * Folder to create in tmp folder.
     */
    public static final String DEFAULT_FOLDER = "EE-Deployer";

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(UnpackDeployableHelper.class);

    /**
     * Utility class, no public constructor.
     */
    private UnpackDeployableHelper() {

    }

    /**
     * Unpack the given archive in a temp folder, then build a local deployable
     * (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the Deployable can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable) throws DeployerException {
        return unpack(deployable, DEFAULT_FOLDER);
    }

    /**
     * @param <T> an object implementing IDeployable
     * @param pattern the path in the pattern to use in the temp folder
     * @param deployable the archive to unpack.
     * @return a new tmp folder
     */
    private static <T extends IDeployable<T>> File createTmpFolder(final String pattern, final T deployable) {
        // Build a temp directory (which is root of all unpack)
        File rootFolder = new File(System.getProperty("java.io.tmpdir") + File.separator + pattern + "-"
                + System.getProperty("user.name", "default"));
        rootFolder.mkdirs();

        // Create EAR directory in this folder (if it is not existing) for
        // unpacking EAR
        File earFolder = new File(rootFolder, deployable.getClass().getSimpleName());
        earFolder.mkdirs();
        return earFolder;
    }

    /**
     * Unpack the given archive in a temp folder, then build a local deployable
     * (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param pattern the path in the pattern to use in the temp folder
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the Deployable can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final String pattern) throws DeployerException {
        return unpack(deployable, createTmpFolder(pattern, deployable), null);
    }

    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param folder the path of the folder for unpacking the archive
     * @param archiveName the name to use for the unpacked archived
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the Deployable can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File folder, final String archiveName)
            throws DeployerException {
        return unpack(deployable, folder, archiveName, false);
    }


    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param folder the path of the folder for unpacking the archive
     * @param archiveName the name to use for the unpacked archived
     * @param keepExistingFiles if 'true', do not override existing files
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the Deployable can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File folder, final String archiveName,
            final boolean keepExistingFiles) throws DeployerException {
        return unpack(deployable, folder, archiveName, keepExistingFiles, null);
    }


    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param folder the path of the folder for unpacking the archive
     * @param archiveName the name to use for the unpacked archived
     * @param keepExistingFiles if 'true', do not override existing files
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the Deployable can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File folder, final String archiveName,
            final boolean keepExistingFiles, final IDeployableHelper deployableHelper) throws DeployerException {

        // Name of the unpacked EAR
        File unpackedFolder = null;

        if (archiveName == null) {
            // Now, create a directory with the name of the given deployable
            URL url = null;
            try {
                url = deployable.getArchive().getURL();
            } catch (ArchiveException e) {
                throw new DeployerException("Cannot get the URL for the deployable '" + deployable + "'.", e);
            }

            // extract filename from the URL for creating the directory
            String fileName = URLUtils.shorterName(url);
            unpackedFolder = new File(folder, fileName);
        } else {
            unpackedFolder = new File(folder, archiveName);
        }

        unpackedFolder.mkdirs();
        return unpack(deployable, unpackedFolder, keepExistingFiles, deployableHelper);

    }

    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param unpackedFolder the path of the folder for unpacking the archive
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the EAR can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File unpackedFolder) throws DeployerException {
        return unpack(deployable, unpackedFolder, false);
    }

    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param unpackedFolder the path of the folder for unpacking the archive
     * @param keepExistingFiles if 'true', do not override existing files
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the EAR can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File unpackedFolder,
            final boolean keepExistingFiles) throws DeployerException {
        return unpack(deployable, unpackedFolder, keepExistingFiles, (IDeployableHelper) null);
    }

    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param unpackedFolder the path of the folder for unpacking the archive
     * @param keepExistingFiles if 'true', do not override existing files
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the EAR can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File unpackedFolder,
            final boolean keepExistingFiles, final IDeployableHelper deployableHelper) throws DeployerException {
        return unpack(deployable, unpackedFolder, keepExistingFiles, null, deployableHelper);
    }



    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param pattern the path in the pattern to use in the temp folder
     * @param keepExistingFiles if 'true', do not override existing files
     * @param earUnpackOpts options to unpack an ear
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the Deployable can't be unpacked
     */
    public static <T extends IDeployable<T>> T unpack(final T deployable, final String pattern,
            final boolean keepExistingFiles, final EarUnpackOpts earUnpackOpts) throws DeployerException {
        File unpackedFolder = new File(createTmpFolder(pattern, deployable), deployable.getShortName());
        unpackedFolder.mkdir();
        return unpack(deployable, unpackedFolder, keepExistingFiles, earUnpackOpts);
    }

    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param unpackedFolder the path of the folder for unpacking the archive
     * @param keepExistingFiles if 'true', do not override existing files
     * @param earUnpackOpts options to unpack an ear
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the EAR can't be unpacked
     */
    @SuppressWarnings("unchecked")
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File unpackedFolder,
            final boolean keepExistingFiles, final EarUnpackOpts earUnpackOpts)
            throws DeployerException {
        return unpack(deployable, unpackedFolder, keepExistingFiles, earUnpackOpts, null);
    }


    /**
     * Unpack the given archive in the given folder, then build a local
     * deployable (and fill it with submodules for EAR) and then return it.
     * @param deployable the archive to unpack.
     * @param unpackedFolder the path of the folder for unpacking the archive
     * @param keepExistingFiles if 'true', do not override existing files
     * @param earUnpackOpts options to unpack an ear
     * @param <T> an object implementing IDeployable
     * @return a new deployable (which is unpacked)
     * @throws DeployerException if the EAR can't be unpacked
     */
    @SuppressWarnings("unchecked")
    public static <T extends IDeployable<T>> T unpack(final T deployable, final File unpackedFolder,
            final boolean keepExistingFiles, final EarUnpackOpts earUnpackOpts, final IDeployableHelper deployableHelper)
            throws DeployerException {

        // Path to the unpacked deployable
        T unpackedDeployable = null;

        // Create new deployable
        IArchive builtArchive = ArchiveManager.getInstance().getArchive(unpackedFolder);

        // Now, do a dump of each resource of the given Deployable and do a dump
        // in the
        // local folder
        Iterator<URL> itResouces = null;
        try {
            itResouces = deployable.getArchive().getResources();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get the resources on the archive '" + deployable.getArchive() + "'.", e);
        }

        URL deployableURL = null;
        try {
            deployableURL = deployable.getArchive().getURL();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get the resources on the archive '" + deployable.getArchive() + "'.", e);
        }

        // Dump all resources
        List<File> entries = new ArrayList<File>();
        while (itResouces.hasNext()) {
            // Dump the input stream of the URL to the local folder
            URL url = itResouces.next();

            // Switch (contains !)
            String entryName = null;
            if (url.getPath().contains("!")) {
                entryName = url.getPath().substring(url.getPath().lastIndexOf("!") + 2);
            } else {
                entryName = url.getPath().substring(deployableURL.getPath().length());
            }

            // unpack file
            File entryFile = new File(unpackedFolder, entryName);
            logger.debug("Dumping url ''{0}'' inputstream into ''{1}''", url, entryFile);
            // Create directory (if some of them are missing)
            entryFile.getParentFile().mkdirs();

            // directory, not a file
            if (url.toString().endsWith("/")) {
                entryFile.mkdirs();
                continue;
            }

            // Dump the file

            URLConnection urlConnection = null;
            try {
                urlConnection = url.openConnection();
            } catch (IOException e) {
                throw new DeployerException("Cannot open the connection on the URL '" + url + "'.", e);
            }
            urlConnection.setDefaultUseCaches(false);
            InputStream is = null;
            try {
                is = urlConnection.getInputStream();
            } catch (IOException e) {
                throw new DeployerException("Cannot get the input stream on the URL connection '" + urlConnection + "'.", e);
            }

            try {
                // Do not override existing file depending on the method
                // parameter
                if (!entryFile.exists() || !keepExistingFiles) {
                    FileUtils.dump(is, entryFile);
                }

                entries.add(entryFile);
            } catch (FileUtilsException e) {
                throw new DeployerException("Cannot dump the inputstream of url '" + url + "' into file '" + entryFile + "'.",
                        e);
            } finally {
                try {
                    is.close();
                } catch (IOException e) {
                    logger.warn("Problem when closing the input stream on url ''{0}''", url, e);
                }
            }
        }

        // Get deployable
        try {
            if (deployableHelper == null) {
                unpackedDeployable = (T) DeployableHelper.getDeployable(builtArchive);
            } else {
                unpackedDeployable = (T) deployableHelper.getDeployable(builtArchive);
            }
        } catch (DeployableHelperException e) {
            throw new DeployerException("Cannot get Deployable on the archive '" + builtArchive + "'", e);
        }
        // Do the link with original and unpack file
        unpackedDeployable.setOriginalDeployable(deployable);
        deployable.setUnpackedDeployable(unpackedDeployable);

        // EAR Case
        if (EARDeployableImpl.class.isAssignableFrom(deployable.getClass())) {
            // Cast the unpacked if ok
            if (!EARDeployableImpl.class.isAssignableFrom(unpackedDeployable.getClass())) {
                throw new DeployerException("The unpacked version of the deployable '" + deployable
                        + "' is not an EAR. Type is '" + unpackedDeployable + "'.");
            }
            EARDeployableImpl earDeployableImpl = EARDeployableImpl.class.cast(unpackedDeployable);

            // If there is info (deployment desc), add files found in this info
            // object in the given order
            IEARInfo earInfo = earDeployableImpl.getEARInfo();
            List<File> analyzedFiles = new ArrayList<File>();
            if (earInfo != null) {
                List<Module> modules = earInfo.getModules();
                if (modules != null) {
                    for (Module module : modules) {
                        File file = new File(unpackedFolder, module.getPath());
                        addFileInDeployable(file, earDeployableImpl, earUnpackOpts);

                        // File already analyzed
                        analyzedFiles.add(file);
                    }
                }
            }

            // Add remaining files
            for (File entryFile : entries) {
                if (analyzedFiles.contains(entryFile)) {
                    continue;
                }

                // If there is an application.xml file, all other files are part of the type "unknown"
                // Do not try to detect type of archive
                if (earInfo != null) {

                    // Add files as unknown deployable
                    // Get archive
                    IArchive tmpArchive = ArchiveManager.getInstance().getArchive(entryFile);

                    // Valid archive, then get the type of the deployable
                    // disallow File Archives in EAR
                    if ((tmpArchive != null) && !(tmpArchive instanceof IFileArchive)) {
                        UnknownDeployable unknownDeployable = new UnknownDeployableImpl(tmpArchive);
                        // Add the deployable into the EAR deployable
                        earDeployableImpl.addDeployable(unknownDeployable, earUnpackOpts);
                    }
                } else {
                    addFileInDeployable(entryFile, earDeployableImpl, earUnpackOpts);
                }
            }
            earDeployableImpl.resolveClassPath();
        }

        return unpackedDeployable;
    }

    /**
     * Analyze the given file by finding its deployable object and add it to the
     * given EAR deployable.
     * @param file the file to analyze
     * @param earDeployableImpl the deployable on which the new deployable will
     *        be added
     * @param earUnpackOpts options to unpack an ear
     * @throws DeployerException if deployable can't be added
     */
    protected static void addFileInDeployable(final File file, final EARDeployableImpl earDeployableImpl,
            final EarUnpackOpts earUnpackOpts) throws DeployerException {
        addFileInDeployable(file, earDeployableImpl, earUnpackOpts, null);
    }

    /**
     * Analyze the given file by finding its deployable object and add it to the
     * given EAR deployable.
     * @param file the file to analyze
     * @param earDeployableImpl the deployable on which the new deployable will
     *        be added
     * @param earUnpackOpts options to unpack an ear
     * @throws DeployerException if deployable can't be added
     */
    protected static void addFileInDeployable(final File file, final EARDeployableImpl earDeployableImpl,
            final EarUnpackOpts earUnpackOpts, IDeployableHelper deployableHelper) throws DeployerException {
        // Get archive
        IArchive tmpArchive = ArchiveManager.getInstance().getArchive(file);

        // Valid archive, then get the type of the deployable
        // disallow File Archives in EAR
        if ((tmpArchive != null) && !(tmpArchive instanceof IFileArchive)) {
            // Analyze the new file
            IDeployable<?> tmpDeployable = null;
            try {
                // Disable bundle inside an EAR if they're not in a specific
                // directory
                if (deployableHelper == null) {
                    tmpDeployable = DeployableHelper.getDeployable(tmpArchive, false);
                } else {
                    tmpDeployable = deployableHelper.getDeployable(tmpArchive, false);
                }

            } catch (DeployableHelperException e) {
                throw new DeployerException("Cannot get a deployable on the file '" + file + "'.", e);
            }
            // Add the deployable into the EAR deployable
            earDeployableImpl.addDeployable(tmpDeployable, earUnpackOpts);
        }
    }

}
