/**
 * EasyBeans
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PackDeployableHelper.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.helper;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Iterator;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.ee.deploy.api.deployable.EARDeployable;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.url.URLUtils;

/**
 * Allow to pack a given deployable.
 * @author Florent BENOIT
 */
public final class PackDeployableHelper {

    /**
     * Buffer SIZE.
     */
    private static final int BUFFER = 4096;

    /**
     * Utility class, no public constructor.
     */
    private PackDeployableHelper() {

    }

    /**
     * Pack the given deployable.
     * @param deployable the archive to pack.
     * @param folder the path of the folder for packing the archive
     * @param archiveName the name of the archive to create.
     * @param <T> an object implementing IDeployable
     * @return a new JAR File (which is packed)
     * @throws DeployerException if the Deployable can't be packed
     */
    @SuppressWarnings("unchecked")
    public static <T extends IDeployable<T>> File pack(final T deployable, final File folder, final String archiveName)
            throws DeployerException {

        T workingDeployable = deployable;
        EARDeployable earworkingDeployable = null;
        // Deployable is unpacked or not ?
        T unpackedDeployable = deployable.getUnpackedDeployable();
        if (unpackedDeployable != null) {
            workingDeployable = unpackedDeployable;
        }
        boolean isEar = false;

        // Warning: workingDeployable instanceof EARDeployable
        // is not compilable with Sun JDK
        if (EARDeployable.class.isAssignableFrom(workingDeployable.getClass())) {
            isEar = true;
            // neither (EARDeployable) workingDeployable
            earworkingDeployable = EARDeployable.class.cast(workingDeployable);
        }

        // Root URL of the unpacked deployable
        URL archiveURL = null;
        try {
            archiveURL = workingDeployable.getArchive().getURL();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get URL for the given archive", e);
        }

        // Now, create a file and store the stream of the given deployable into this file
        Iterator<URL> itResources = null;
        try {
            itResources = workingDeployable.getArchive().getResources();
        } catch (ArchiveException e) {
           throw new DeployerException("Cannot get the resources for the deployable '" + workingDeployable + "'.", e);
        }

        // Create a file
        File file = new File(folder, archiveName);

        // File Stream
        FileOutputStream fileOS = null;
        try {
            fileOS = new FileOutputStream(file);
        } catch (FileNotFoundException e) {
            throw new DeployerException("Cannot create file output stream on the file '" + file + "'", e);
        }

        // Jar stream
        JarOutputStream fileJarOutputStream = null;
        try {
            fileJarOutputStream = new JarOutputStream(fileOS);
        } catch (IOException e) {
           throw new DeployerException("Cannot create Jar OutputStream", e);
        }

        try {

            int n;
            byte[] buffer = new byte[BUFFER];

            // For each resource, store it in the file
            while (itResources.hasNext()) {

                URL urlResource = itResources.next();
                // jar entry name to create in the current packed archive
                String entryName = urlResource.toExternalForm().substring(archiveURL.toExternalForm().length());
                if (isEar) {
                    // in case of ear some sub archives may be unpacked and modified
                    // such archives must be packed and inserted in the resulting packed ear
                    IDeployable subDeployable = earworkingDeployable.getDeployable(urlResource);
                    if (subDeployable != null) {
                        IDeployable unpackedSubDeployable = (IDeployable) subDeployable.getUnpackedDeployable();
                        if (unpackedSubDeployable != null) {
                            try {
                                URL archiveURLtoPack = unpackedSubDeployable.getArchive().getURL();
                                String san = URLUtils.shorterName(archiveURLtoPack);
                                File packedJar = pack(unpackedSubDeployable, folder, "new-" + san);
                                urlResource = URLUtils.fileToURL(packedJar);
                            } catch (ArchiveException ae) {
                                throw new DeployerException("Cannot pack the deployable '" + deployable + "'", ae);
                            }
                        }
                    }
                }

                // Create entry
                JarEntry jarEntry = new JarEntry(entryName);

                // Store entry
                fileJarOutputStream.putNextEntry(jarEntry);

                // Get an URL connection
                URLConnection urlConnection = urlResource.openConnection();
                urlConnection.setDefaultUseCaches(false);
                InputStream is = urlConnection.getInputStream();

                n = is.read(buffer);
                while (n > 0) {
                    fileJarOutputStream.write(buffer, 0, n);
                    n = is.read(buffer);
                }
                is.close();
                fileJarOutputStream.closeEntry();

            }
        } catch (IOException ioe) {
            throw new DeployerException("Cannot pack the deployable '" + deployable + "'", ioe);
        } finally {
            // Close streams
            try {
                fileJarOutputStream.close();
            } catch (IOException e) {
                throw new DeployerException("Cannot close jar file output stream", e);
            }

            try {
                fileOS.close();
            } catch (IOException e) {
                throw new DeployerException("Cannot close file output stream", e);
            }

        }

        return file;
    }

}
