/**
 * EasyBeans
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ModifyDeployableHelper.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.helper;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.List;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.file.FileUtils;
import org.ow2.util.file.FileUtilsException;
import org.ow2.util.url.URLUtils;

/**
 * Allow to add entries in a given deployable.
 * @author Florent BENOIT
 */
public final class ModifyDeployableHelper {

    /**
     * Folder to create in tmp folder.
     */
    public static final String DEFAULT_FOLDER = "EE-ModifierDeployer";

    /**
     * Utility class has no public constructor.
     */
    private ModifyDeployableHelper() {

    }


    /**
     * Add given entries in the given deployable.
     * @param deployable the archive to modify.
     * @param entries a list of entries to add in the given archive
     * @param <T> an object implementing IDeployable
     * @throws DeployerException if the Deployable can't be modified
     */
    public static <T extends IDeployable<T>> void addInDeployable(final T deployable, final List<DeployableEntry> entries)
            throws DeployerException {
        // Use a default temp directory (which is root of all changes)
        File rootFolder = new File(System.getProperty("java.io.tmpdir") + File.separator + DEFAULT_FOLDER + "-"
                + System.getProperty("user.name", "default"));

        addInDeployable(deployable, entries, rootFolder);
    }

    /**
     * Add given entries in the given deployable.
     * @param deployable the archive to modify.
     * @param entries a list of entries to add in the given archive
     * @param workingFolder a folder used for the work
     * @param <T> an object implementing IDeployable
     * @throws DeployerException if the Deployable can't be modified
     */
    public static <T extends IDeployable<T>> void addInDeployable(final T deployable, final List<DeployableEntry> entries,
            final File workingFolder) throws DeployerException {
        // Build directory
        workingFolder.mkdirs();

        // Deployable used for the changes
        T workDeployable = deployable;

        // Already unpacked ?
        T unpackedDeployable = deployable.getUnpackedDeployable();
        if (unpackedDeployable == null) {
            // Get the URL of the archive
            URL tmpURL = null;
            try {
                tmpURL = deployable.getArchive().getURL();
            } catch (ArchiveException e) {
                throw new DeployerException("Cannot get URL on deployable '" + deployable + "'");
            }
            //Name of the archive
            String archiveName = URLUtils.shorterName(tmpURL);

            // Need to unpack it in a given folder
            unpackedDeployable = UnpackDeployableHelper.unpack(workDeployable, workingFolder, archiveName);

            // Associate
            deployable.setUnpackedDeployable(unpackedDeployable);
            unpackedDeployable.setOriginalDeployable(deployable);
        }

        // If it's unpacked, it should be in a filesystem in a directory
        URL urlDeployable = null;
        try {
            urlDeployable = unpackedDeployable.getArchive().getURL();
        } catch (ArchiveException e) {
           throw new DeployerException("Cannot get URL for the deployable '" + unpackedDeployable + "'.", e);
        }

        // file ?
        if (!"file".equals(urlDeployable.getProtocol())) {
            throw new DeployerException("Protocol for unpacked deployable '" + unpackedDeployable + "' is not a file");
        }

        // Get Directory
        File unpackedDir = URLUtils.urlToFile(urlDeployable);
        if (!unpackedDir.isDirectory()) {
            throw new DeployerException("The file '" + unpackedDir + "' is not a directory");
        }

        // Do nothing if there are no entries
        if (entries == null) {
            return;
        }

        // For each entry, add it in the deployable
        for (DeployableEntry depEntry : entries) {
            // Get Name
            String entryName = depEntry.getName();
            // Get File
            File f = depEntry.getFile();

            // Read stream
            InputStream is;
            try {
                is = new FileInputStream(f);
            } catch (FileNotFoundException e) {
                throw new DeployerException("Cannot get stream on file '" + f + "'", e);
            }

            // Compute path for writing file
            File writingFile = new File(unpackedDir, entryName.replace("/", File.separator));
            writingFile.getParentFile().mkdirs();

            // Write stream
            try {
                FileUtils.dump(is, writingFile);
            } catch (FileUtilsException e) {
                e.printStackTrace();
                throw new DeployerException("Cannot dump stream of file '" + f + "'", e);
            } finally {
                // close stream
                if (is != null) {
                    try {
                        is.close();
                    } catch (IOException e) {
                        throw new DeployerException("Cannot close stream of file '" + f + "'", e);
                    }
                }
            }
        }
    }

}
