/**
 * OW2 Util
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployableHelper.java 6057 2011-11-06 16:40:41Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.helper;

import static org.ow2.util.ee.deploy.api.deployable.ArchiveType.CLIENT_APP;
import static org.ow2.util.ee.deploy.api.deployable.ArchiveType.EAR;
import static org.ow2.util.ee.deploy.api.deployable.ArchiveType.EJB21JAR;
import static org.ow2.util.ee.deploy.api.deployable.ArchiveType.EJB3JAR;
import static org.ow2.util.ee.deploy.api.deployable.ArchiveType.RAR;
import static org.ow2.util.ee.deploy.api.deployable.ArchiveType.UNKNOWN;
import static org.ow2.util.ee.deploy.api.deployable.ArchiveType.WAR;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Iterator;
import java.util.jar.Attributes;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.archive.api.IArchiveMetadata;
import org.ow2.util.archive.api.IFileArchive;
import org.ow2.util.asm.ClassReader;
import org.ow2.util.ee.deploy.api.deployable.ArchiveType;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.helper.DeployableHelperException;
import org.ow2.util.ee.deploy.impl.deployable.CARDeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.EARDeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.EJB21DeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.EJB3DeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.OSGiDeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.RARDeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.UnknownDeployableImpl;
import org.ow2.util.ee.deploy.impl.deployable.WARDeployableImpl;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.plan.deploy.deployable.api.FileDeployable;
import org.ow2.util.plan.deploy.deployable.api.factory.FileDeployableException;
import org.ow2.util.plan.deploy.deployable.api.factory.IFileDeployableFactory;
import org.ow2.util.xml.DocumentParser;
import org.ow2.util.xml.DocumentParserException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Analyze an archive and build the associated Deployable object.<br>
 * For an .ear file the EARDeployable object will be returned.
 * @author Florent Benoit
 */
public final class DeployableHelper {

    /**
     * EAR extension.
     */
    private static final String EAR_EXTENSION = ".ear";

    /**
     * WAR extension.
     */
    private static final String WAR_EXTENSION = ".war";

    /**
     * RAR extension.
     */
    private static final String RAR_EXTENSION = ".rar";

    /**
     * JAR extension.
     */
    private static final String JAR_EXTENSION = ".jar";

    /**
     * EJB XML DD.
     */
    private static final String EJB_DD = "META-INF/ejb-jar.xml";

    /**
     * Web XML DD.
     */
    private static final String WAR_DD = "WEB-INF/web.xml";

    /**
     * Rar XML DD.
     */
    private static final String RAR_DD = "META-INF/ra.xml";

    /**
     * EAR XML DD.
     */
    private static final String EAR_DD = "META-INF/application.xml";

    /**
     * APP Client XML DD.
     */
    private static final String CAR_DD = "META-INF/application-client.xml";

    /**
     * Java EE namespace (Java EE).
     */
    private static final String JAVAEE_NS = "http://java.sun.com/xml/ns/javaee";

    /**
     * Required entry for Bundle in Manifest header.
     */
    private static final String BUNDLE_SYMBOLIC_NAME = "Bundle-SymbolicName";

    /**
     * Required entry for WAB Bundle in Manifest header.
     */
    private static final String WEB_CONTEXTPATH = "Web-ContextPath";

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(DeployableHelper.class);

    /**
     * If OSGi mode is enabled, detect bundles.
     */
    private boolean isOSGiEnabled = true;

    /**
     * Archive used by this helper.
     */
    private IArchive archive = null;

    /**
     * The file deployable factory used for FileArchive objects.
     */
    private static IFileDeployableFactory fileDeployableFactory = null;

    /**
     * Utility class, no public constructor.
     * @param archive the given archive
     */
    private DeployableHelper(final IArchive archive) {
        this.archive = archive;
    }

    /**
     * Gets the Deployable object for the given URL.
     * @param archive the given archive
     * @return the Deployable object
     * @throws DeployableHelperException if the analyze fails
     */
    public static IDeployable<?> getDeployable(final IArchive archive) throws DeployableHelperException {
        return getDeployable(archive, true);
    }

    /**
     * Gets the Deployable object for the given URL.
     * @param archive the given archive
     * @param isOSGiEnabled if OSGi Deployable can be used
     * @return the Deployable object
     * @throws DeployableHelperException if the analyze fails
     */
    public static IDeployable<?> getDeployable(final IArchive archive, final boolean isOSGiEnabled)
            throws DeployableHelperException {
        long t = System.currentTimeMillis();
        // Build object
        DeployableHelper helper = new DeployableHelper(archive);
        helper.isOSGiEnabled = isOSGiEnabled;

        // Analyze the archive and return the given deployable
        try {
            return helper.analyze();
        } finally {
            logger
                    .debug("Detect type for ''{0}'' in ''{1}'' ms", archive.getName(), Long.valueOf(System.currentTimeMillis()
                            - t));
        }

    }

    /**
     * Analyze the URL and create a deployable object.
     * @return the deployable object
     * @throws DeployableHelperException if the analyze fails
     */
    private IDeployable<?> analyze() throws DeployableHelperException {

        /**
         *  First, handle single file like single XML file (no JAR file or Directory-JAR)
         */
        if (archive instanceof IFileArchive) {
            // Factory for this type is available ?
            if (DeployableHelper.fileDeployableFactory != null) {
                FileDeployable<?, ?> fileDeployable = null;
                try {
                    fileDeployable = DeployableHelper.fileDeployableFactory.getFileDeployable((IFileArchive) archive);
                } catch (FileDeployableException e) {
                    logger.debug("It's not a recognized file deployable", e);
                }
                if (fileDeployable != null) {
                    return fileDeployable;
                }
            }
            // Not be able to detect the deployable for this File archive
            return new UnknownDeployableImpl(archive);
        }

        /**
         *  Next, handle other kind of Archive (JarArchive or Directory archive)
         */

        // OSGi aware modules (if enabled)
        if (isOSGiEnabled) {

            // Read symbolic name
            String symbolicName = null;
            IArchiveMetadata archiveMetadata = archive.getMetadata();
            if (archiveMetadata != null) {
                // Required attribute Bundle-Symbolic-Name is present ?
                symbolicName = archiveMetadata.get(BUNDLE_SYMBOLIC_NAME);
            }

            // if symbolic-name is here, check the type of the archive
            if (symbolicName != null) {

                // JAR file with OSGi metadata = bundle
                if (archive.getName().toLowerCase().endsWith(JAR_EXTENSION)) {
                    return new OSGiDeployableImpl(archive);
                }

                // WAR file with OSGi metadata, check that it is a compliant WAB
                String webContextPath = archiveMetadata.get(WEB_CONTEXTPATH);
                if (archive.getName().toLowerCase().endsWith(WAR_EXTENSION)) {
                    if (webContextPath != null && !webContextPath.equals("")) {
                        return new OSGiDeployableImpl(archive);
                    } else {
                        logger.warn("Archive ''{0}'' has OSGi metadata but is not compliant with Web Application "
                                + "Bundle specification. (No Web-ContextPath attribute)"
                                + " Thus it is not detected as an OSGi bundle." + " OSGi metadata should be fixed or removed.",
                                archive);
                    }
                }
            }
        }

        /**
         *  Now, we check other kind of modules
         */
        // more doc on the Java EE 5.0 specification chapter EE 8.4.2
        // First, try to detect if the archive has a Deployment descriptor (easy
        // case).
        logger.debug("Try to read XML DD in order to find the type of the archive ''{0}''", archive.getName());

        switch (getXMLType()) {

        /**
         * Found META-INF/ejb-jar.xml (version ejb 2.1).
         */
        case EJB21JAR:
            logger.debug("''{0}'' is an EJB 2.1 with XML DD", archive);
            return new EJB21DeployableImpl(archive);

        /**
         * Found META-INF/ejb-jar.xml (version ejb 3.0).
         */
        case EJB3JAR:
            logger.debug("''{0}'' is an EJB 3 with XML DD", archive);
            return new EJB3DeployableImpl(archive);

        /**
         * Found WEB-INF/web.xml.
         */
        case WAR:
            logger.debug("''{0}'' is a WAR with XML DD", archive);
            return new WARDeployableImpl(archive);

        /**
         * Found META-INF/application.xml.
         */
        case EAR:
            logger.debug("''{0}'' is an EAR with XML DD", archive);
            return new EARDeployableImpl(archive);

        /**
         * Found META-INF/ra.xml.
         */
        case RAR:
            logger.debug("''{0}'' is a RAR with XML DD", archive);
            return new RARDeployableImpl(archive);

        /**
         * Found META-INF/application-client.xml.
         */
        case CLIENT_APP:
            logger.debug("''{0}'' is a CAR with XML DD", archive);
            return new CARDeployableImpl(archive);

        // Well, no XML found in the archive, need to detect the type of the
        // archive !
        default:

            // Try with file extension
            logger.debug("No XML DD, try with file extension");

            if (archive.getName().toLowerCase().endsWith(EAR_EXTENSION)) {
                // EAR
                logger.debug("''{0}'' is an EAR", archive);
                return new EARDeployableImpl(archive);
            } else if (archive.getName().toLowerCase().endsWith(WAR_EXTENSION)) {
                // WAR
                logger.debug("''{0}'' is a WAR", archive);
                return new WARDeployableImpl(archive);
            } else if (archive.getName().toLowerCase().endsWith(RAR_EXTENSION)) {
                // RAR
                logger.debug("''{0}'' is a RAR", archive);
                return new RARDeployableImpl(archive);
            }

            // For jar file, it there is a MANIFEST with a Main-Class attribute, it is a Client module
            if (archive.getName().toLowerCase().endsWith(JAR_EXTENSION)) {
                IArchiveMetadata archiveMetadata = archive.getMetadata();
                String mainClass = null;
                if (archiveMetadata != null) {
                    mainClass = archiveMetadata.get(Attributes.Name.MAIN_CLASS.toString());
                    if (mainClass != null) {
                        logger.debug("Detecting Main-Class attribute in archive ''{0}'' with value ''{1}''", archive
                                .getName(), mainClass);
                        return new CARDeployableImpl(archive);
                    }
                }
            }
            // Not able to detect, run class analyzer
            return classDetect();
        }
    }

    /**
     * Analyze the archive to see if there are some Annotated classes that can find the type of the archive.
     * @return a deployable object
     * @throws DeployableHelperException if the analyze of the classes is failing.
     */
    private IDeployable<?> classDetect() throws DeployableHelperException {

        // Scan all classes of the given archive but break as soon as a matching annotation has been found
        Iterator<URL> iterator;
        ArchiveType detectedType = null;

        // Use a specialized visitor
        DeployableDetectVisitor detectVisitor = new DeployableDetectVisitor();

        try {
            iterator = archive.getResources();
            // Scan all resources and stop if type has been found
            while (iterator.hasNext() && detectedType == null) {
                URL url = iterator.next();
                // only .class
                if (url.getPath().toLowerCase().endsWith(".class")) {
                    try {
                        URLConnection urlConnection = url.openConnection();
                        urlConnection.setDefaultUseCaches(false);
                        InputStream is = urlConnection.getInputStream();

                        try {
                            new ClassReader(is).accept(detectVisitor, 0);
                        } catch (RuntimeException e) {
                            // ASM may produce Runtime Exception
                            throw new DeployableHelperException("Error while analyzing file entry '" + url + "' in archive '"
                                    + archive.getName() + "'", e);
                        } finally {
                            if (is != null) {
                                is.close();
                            }
                        }

                        detectedType = detectVisitor.getArchiveType();
                    } catch (IOException ioe) {
                        throw new DeployableHelperException("Error while analyzing file entry '" + url + "' in archive '"
                                + archive.getName() + "'", ioe);
                    }

                }
            }

        } catch (ArchiveException e) {
            throw new DeployableHelperException("Error while analyzing archive '" + archive.getName() + "'", e);
        } finally {
            archive.close();
        }

        // Found EJB3 annotations ?
        if (ArchiveType.EJB3JAR == detectedType) {
            return new EJB3DeployableImpl(archive);
        }

        // Not found the type !
        return new UnknownDeployableImpl(archive);
    }


    /**
     * Try to see if there is an XML file in the archive.
     * If it is an EJB XML file, try to get the version.
     * @return the type of the archive.
     * @throws DeployableHelperException if the archive can't be analyzed
     */
    private ArchiveType getXMLType() throws DeployableHelperException {

        // Get entry ?
        try {
            if (archive.getResource(EAR_DD) != null) {
                return EAR;
            } else if (archive.getResource(WAR_DD) != null) {
                return WAR;
            } else if (archive.getResource(CAR_DD) != null) {
                return CLIENT_APP;
             } else if (archive.getResource(RAR_DD) != null) {
                 return RAR;
             } else if (archive.getResource(EJB_DD) != null) {
                 // close archive
                 archive.close();

                 // Needs to analyze the version of the XML
                 return ejbXMLType();
             }
        } catch (ArchiveException e) {
            throw new DeployableHelperException("Cannot analyze archive '" + archive.getName() + "' for finding XML DD entry", e);
        } finally {
            archive.close();
        }

        // No XML DD found
        return UNKNOWN;

    }

    /**
     * Know that the archive is an EJB but needs to analyze the XML in order to see if it is a 2.1 or 3 EJB.
     * @return EJB 2 or 3 type
     * @throws DeployableHelperException if the archive can't be analyzed
     */
    private ArchiveType ejbXMLType() throws DeployableHelperException {

        // Get the URL
        URL ejbDD = null;
        try {
            ejbDD = archive.getResource(EJB_DD);
        } catch (ArchiveException e) {
            throw new DeployableHelperException("Cannot get the '" + EJB_DD + "' entry in the archive '" + archive.getName()
                    + "'.", e);
        } finally {
            archive.close();
        }

        // Get document without validation
        Document document = null;
        try {
            document = DocumentParser.getDocument(ejbDD, false, null);
        } catch (DocumentParserException e) {
            throw new DeployableHelperException("Cannot parse the url", e);
        }

        // Root element = <ejb-jar>
        Element ejbJarRootElement = document.getDocumentElement();

        // get the namespace of this element
        String ns = ejbJarRootElement.getNamespaceURI();

        // Only EJB3 have this namespace
        if (JAVAEE_NS.equals(ns)) {
            return EJB3JAR;
        }

        // else if it is a previous version
        return EJB21JAR;
    }

    /**
     * Sets the file deployable factory.
     * @param factory the new file deployable factory.
     */
    public static void setFileDeployableFactory(final IFileDeployableFactory factory) {
        fileDeployableFactory = factory;
    }

    /**
     * Returns the file deployable factory.
     * @return the file deployable factory.
     */
    public static IFileDeployableFactory getFileDeployableFactory() {
        return fileDeployableFactory;
    }

}
