/**
 * EasyBeans
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployerManager.java 6234 2012-04-13 14:32:30Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.deployer;

import java.util.ArrayList;
import java.util.List;

import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployer.IDeployer;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManager;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManagerCallback;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManagerReportCallback;
import org.ow2.util.ee.deploy.api.deployer.UnsupportedDeployerException;
import org.ow2.util.ee.deploy.api.report.IDeploymentReport;
import org.ow2.util.ee.deploy.impl.report.DeploymentReport;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This manager is managing different deployers. The deployer that will be used
 * is a deployer that is supporting a given IDeployable object. If there are
 * many deployers for the same type of Deployable, the first deployer that is
 * supporting the given deployable will be used.
 * At each deployment/undeployment phase, invoke registered callbacks.
 * @author Florent Benoit
 * @author Francois Fornaciari (Callback support)
 * @author Jeremy Cazaux (deployment/undeployment of a list of deployables)
 */
public final class DeployerManager implements IDeployerManager {

    /**
     * Name of the property thats defines the deployer's class. (optional)
     */
    public static final String DEPLOYER_FACTORY_CLASS_NAME = DeployerManager.class.getName();

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(DeployerManager.class);

    /**
     * List of deployers.
     */
    private List<IDeployer> deployers = null;

    /**
     * List of callbacks to invoke.
     */
    private List<IDeployerManagerCallback> callbacks = null;

    /**
     * List of {@link IDeployerManagerReportCallback} to invoke
     */
    private List<IDeployerManagerReportCallback> reportCallbacks;

    /**
     * Create a new Manager of deployer.
     */
    public DeployerManager() {
        this.deployers = new ArrayList<IDeployer>();
        this.callbacks = new ArrayList<IDeployerManagerCallback>();
        this.reportCallbacks = new ArrayList<IDeployerManagerReportCallback>();
    }


    /**
     * Register a new Deployer on this manager instance.
     * @param deployer the deployer to register
     */
    public void register(final IDeployer deployer) {
        if (deployer != null) {
            logger.debug("Registering deployer ''{0}''", deployer.getClass());
            deployers.add(deployer);
        }
    }

    /**
     * Unregister a Deployer from this manager instance.
     * @param deployer the given deployer to unregister.
     */
    public void unregister(final IDeployer deployer) {

        if (deployers.contains(deployer)) {
            logger.debug("Unregistering deployer ''{0}''", deployer);
            deployers.remove(deployer);
        } else {
            logger.warn("Cannot unregister the deployer ''{0}'' as it is not registered.", deployer);
        }
    }

    /**
     * Find the first available deployer that is supporting the given deployable.
     * @param deployable the given deployable to test.
     * @return the fist available deployer
     * @throws UnsupportedDeployerException if no deployer is found for the given deployable.
     */
    protected IDeployer findDeployerForDeployable(final IDeployable<?> deployable) throws UnsupportedDeployerException {
        for (IDeployer deployer : deployers) {
            if (deployer.supports(deployable)) {
                return deployer;
            }
        }
        throw new UnsupportedDeployerException("No deployer was found for the deployable '" + deployable + "'.");
    }

    /**
     * Find the first available deployer that is supporting the given deployable.
     * @param deployables the list of deployable to test.
     * @return the fist available deployer
     * @throws UnsupportedDeployerException if no deployer is found for these deployables.
     */
    protected IDeployer findDeployerForDeployables(final List<IDeployable<?>> deployables) throws UnsupportedDeployerException {
        for (IDeployer deployer : deployers) {
            if (deployer.supports(deployables)) {
                return deployer;
            }
        }
        StringBuffer sb = new StringBuffer();
        for (IDeployable deployable: deployables) {
            sb.append("No deployer founded for the deployable '" + deployable + "\n");
        }
        throw new UnsupportedDeployerException(sb.toString());
    }

    /**
     * Deploy a given deployable by finding the first deployer that accept this type of deployable.
     * @param deployable the given deployable to deploy.
     * @throws DeployerException if it cannot be undeployed.
     * @throws UnsupportedDeployerException if no deployer is found for the given deployable.
     */
    public void deploy(final IDeployable<?> deployable) throws DeployerException, UnsupportedDeployerException {
        callback(DeployerManagerCallbackType.PRE_DEPLOY, deployable);
        // find a deployer and deploy
        findDeployerForDeployable(deployable).deploy(deployable);
        callback(DeployerManagerCallbackType.POST_DEPLOY, deployable);
    }

    /**
     * Do a deployment operation with a multi type list of deployable by finding the first deployer that accept
     * each type of deployable.
     * @param deployables the list of deployable to deploy.
     * @param isDeployOperation True if it's a deploy operation. Otherwise it's an undeploy operation
     * @return the list of {@link IDeploymentReport}
     */
    protected List<IDeploymentReport> doDeploymentOperation(final List<IDeployable<?>> deployables,
                                                            final boolean isDeployOperation)  {

        //contains deployables of a same type
        List<IDeployable<?>> deployablesOneTypeList = new ArrayList<IDeployable<?>>();

        //List of deployment reports
        List<IDeploymentReport> deploymentReports = new ArrayList<IDeploymentReport>();

        int i = 0;

        //(un)deploy a multi type list by (un)deploying each one type list without breaking the order of the list
        if (deployables.size() > 0) {

            deployablesOneTypeList.add(deployables.get(0));

            while (i < deployables.size() - 1) {

                if (deployables.get(i).getClass().getName().equals(deployables.get(i + 1).getClass().getName())) {
                    deployablesOneTypeList.add(deployables.get(i + 1));
                } else {

                    if (isDeployOperation) {
                        //deploy the one type list
                        deploymentReports.addAll(deployOneTypeList(deployablesOneTypeList));
                    } else {
                        //else it's an undeploy operation
                        deploymentReports.addAll(undeployOneTypeList(deployablesOneTypeList));
                    }

                    deployablesOneTypeList = new ArrayList<IDeployable<?>>();
                    deployablesOneTypeList.add(deployables.get(i + 1));
                }
                i++;
            }

            if (isDeployOperation) {
                //deploy the one type list
                deploymentReports.addAll(deployOneTypeList(deployablesOneTypeList));
            } else {
                //else it's an undeploy operation
                deploymentReports.addAll(undeployOneTypeList(deployablesOneTypeList));
            }
        }

        return deploymentReports;
    }

    /**
     * {@inheritDoc}
     */
    public List<IDeploymentReport> deploy(final List<IDeployable<?>> deployables) {
        return doDeploymentOperation(deployables, true);
    }

    /**
     * Deploy a one type list of deployable by finding the first deployer that accept
     * this type of deployable.
     * @param deployables the list of deployable to deploy.
     * @return the {@link IDeploymentReport}
     */
    protected List<IDeploymentReport> deployOneTypeList(final List<IDeployable<?>> deployables) {

         //pre deploy callbacks
        for (IDeployable deployable: deployables) {
            reportCallback(DeployerManagerCallbackType.PRE_DEPLOY, deployable);
        }

        // find a deployer and deploy all deployables
        List<IDeploymentReport> deploymentReports = null;
        try {
            deploymentReports = findDeployerForDeployables(deployables).deploy(deployables);
        } catch (UnsupportedDeployerException e) {
            deploymentReports = new ArrayList<IDeploymentReport>();
            for (IDeployable deployable: deployables) {
                DeploymentReport deploymentReport = new DeploymentReport();
                deploymentReport.setDeploymentOk(false);
                deploymentReport.setException(e);
                deploymentReport.setDeployable(deployable);
                deploymentReports.add(deploymentReport);
            }
        }

        //post deploy callbacks
        if (deploymentReports != null) {
            for (IDeploymentReport deploymentReport: deploymentReports) {
                reportCallback(DeployerManagerCallbackType.POST_DEPLOY, deploymentReport);
            }
        }

        return deploymentReports;
    }

    /**
     * Undeploy a one type list of deployable by finding the first deployer that accept
     * this type of deployable.
     * @param deployables the list of deployable to undeploy.
     * @return the {@link IDeploymentReport}
     */
    protected List<IDeploymentReport> undeployOneTypeList(final List<IDeployable<?>> deployables) {

        //pre undeploy callbacks
        for (IDeployable deployable: deployables) {
            reportCallback(DeployerManagerCallbackType.PRE_UNDEPLOY, deployable);
        }

        // find a deployer and undeploy all deployables
        List<IDeploymentReport> deploymentReports;
        try {
            deploymentReports = findDeployerForDeployables(deployables).undeploy(deployables);
        } catch (UnsupportedDeployerException e) {
            deploymentReports = new ArrayList<IDeploymentReport>();
            for (IDeployable deployable: deployables) {
                DeploymentReport deploymentReport = new DeploymentReport();
                deploymentReport.setDeploymentOk(false);
                deploymentReport.setDeployable(deployable);
                deploymentReport.setException(e);
                deploymentReports.add(deploymentReport);
            }
            
        }

        //post undeploy callbacks
        if (deploymentReports != null) {
            for (IDeploymentReport deploymentReport: deploymentReports) {
                reportCallback(DeployerManagerCallbackType.POST_UNDEPLOY, deploymentReport);
            }
        }

        return deploymentReports;
    }

    /**
     * Undeploy a given deployable by finding the first deployer that accept this type of deployable.
     * @param deployable the given deployable to undeploy.
     * @throws DeployerException if it cannot be undeployed.
     * @throws UnsupportedDeployerException if no deployer is found for the given deployable.
     */
    public void undeploy(final IDeployable<?> deployable) throws DeployerException, UnsupportedDeployerException {
        callback(DeployerManagerCallbackType.PRE_UNDEPLOY, deployable);
        // find a deployer and undeploy
        findDeployerForDeployable(deployable).undeploy(deployable);
        callback(DeployerManagerCallbackType.POST_UNDEPLOY, deployable);
    }

    /**
     * {@inheritDoc}
     */
    public List<IDeploymentReport> undeploy(final List<IDeployable<?>> deployables)  {
        return doDeploymentOperation(deployables, false);
    }

    /**
     * {@inheritDoc}
     */
    public List<String> getDeployerClasses() {
        List<String> classes = new ArrayList<String>();
        for (IDeployer deployer: this.deployers) {
            classes.add(deployer.getClass().getName());
        }
        return classes;
    }

    /**
     * Deploy a given deployable by finding the first deployer that accept this type of deployable.
     * @param deployable the given deployable to deploy.
     * @throws DeployerException if it cannot be undeployed.
     * @throws UnsupportedDeployerException if no deployer is found for the given deployable.
     */
    public boolean isDeployed(final IDeployable<?> deployable) throws DeployerException, UnsupportedDeployerException {
        // find a deployer and deploy
        return findDeployerForDeployable(deployable).isDeployed(deployable);
    }

    /**
     * Add a new callback instance that will be invoked at each deployment/undeployment phase.
     * @param callback The given callback
     */
    public void addCallback(final IDeployerManagerCallback callback) {
        if (!callbacks.contains(callback)) {
            callbacks.add(callback);
        } else {
            logger.warn("Callback already added");
        }
    }

    /**
     * Remove a callback instance.
     * @param callback The given callback
     */
    public void removeCallback(final IDeployerManagerCallback callback) {
        if (!callbacks.contains(callback)) {
            logger.warn("Callback doesn't exist");
        } else {
            callbacks.remove(callback);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void addCallback(final IDeployerManagerReportCallback callback) {
        if (!this.reportCallbacks.contains(callback)) {
            this.reportCallbacks.add(callback);
        } else {
            logger.warn("Callback already added");
        }
    }

    /**
     * {@inheritDoc}
     */
    public void removeCallback(final IDeployerManagerReportCallback callback) {
        if (!this.reportCallbacks.contains(callback)) {
            logger.warn("Callback doesn't exist");
        } else {
            this.reportCallbacks.remove(callback);
        }
    }

    /**
     * Perform the right callback depending on the callback type.
     * @param type The callback type
     * @param deployable The deployable to deploy/undeploy
     */
    private void callback(final DeployerManagerCallbackType type, final IDeployable<?> deployable) {
        for (IDeployerManagerCallback callback : callbacks) {
            try {
                switch (type) {
                case PRE_DEPLOY:
                    callback.preDeploy(deployable);
                    break;
                case POST_DEPLOY:
                    callback.postDeploy(deployable);
                    break;
                case PRE_UNDEPLOY:
                    callback.preUndeploy(deployable);
                    break;
                case POST_UNDEPLOY:
                    callback.postUndeploy(deployable);
                    break;
                default:
                    break;
                }
            } catch (Throwable t) {
                logger.error("Error during the execution of the ''{0}'' callback for ''{1}''",
                        type, deployable.getArchive(), t);
                t.printStackTrace();
            }
        }
    }

    /**
     * Perform the right callback depending on the callback type.
     * @param type The callback type
     * @param deploymentReport The {@link IDeploymentReport}
     */
    private void reportCallback(final DeployerManagerCallbackType type, final IDeploymentReport deploymentReport) {
        for (IDeployerManagerReportCallback reportCallback: this.reportCallbacks) {
            try {
                switch (type) {
                    case POST_DEPLOY:
                        reportCallback.postDeploy(deploymentReport);
                        break;
                    case POST_UNDEPLOY:
                        reportCallback.postUndeploy(deploymentReport);
                        break;
                    default:
                        break;
                }
            } catch (Throwable t) {
                logger.error("Error during the execution of the ''{0}'' callback for ''{1}''",
                        type, deploymentReport.getDeployable().getArchive(), t);
                t.printStackTrace();
            }
        }
    }

    /**
     * Perform the right report callback depending on the callback type.
     * @param type The callback type
     * @param deployable The {@link IDeployable}
     */
    private void reportCallback(final DeployerManagerCallbackType type, final IDeployable<?> deployable) {
        for (IDeployerManagerReportCallback reportCallback: this.reportCallbacks) {
            try {
                switch (type) {
                    case PRE_DEPLOY:
                        reportCallback.preDeploy(deployable);
                        break;
                    case PRE_UNDEPLOY:
                        reportCallback.preUndeploy(deployable);
                        break;
                    default:
                        break;
                }
            } catch (Throwable t) {
                logger.error("Error during the execution of the ''{0}'' callback for ''{1}''",
                        type, deployable.getArchive(), t);
                t.printStackTrace();
            }
        }
    }
}
