/**
 * EasyBeans
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AbsDeployer.java 6232 2012-04-13 12:37:03Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.deployer;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployer.UnsupportedDeployerException;
import org.ow2.util.ee.deploy.api.report.IDeploymentReport;
import org.ow2.util.execution.ExecutionResult;
import org.ow2.util.execution.IExecution;
import org.ow2.util.execution.helper.RunnableHelper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.url.URLUtils;

/**
 * Abstract class that defines common methods for deployer.
 * @author Francois Fornaciari
 */
public abstract class AbsDeployer<T extends IDeployable<T>> extends AbsDeployerList<T> {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(AbsDeployer.class);

    /**
     * Map between an URL and the associated Deployable that has been deployed.
     */
    protected Map<URL, IDeployable<T>> deployedDeployables;

    /**
     * Default constructor.
     */
    public AbsDeployer() {
        deployedDeployables = new HashMap<URL, IDeployable<T>>();
    }

    /**
     * {@inheritDoc}
     */
    public abstract void doDeploy(final IDeployable<T> deployable) throws DeployerException;

    /**
     * {@inheritDoc}
     */
    public abstract void doUndeploy(final IDeployable<T> deployable) throws DeployerException;

    /**
     * {@inheritDoc}
     */
    public abstract boolean supports(IDeployable<?> deployable);

    /**
     * {@inheritDoc}
     */
    public void deploy(final IDeployable<T> deployable) throws DeployerException {
        check(deployable);

        RunnableHelper<Void> runner = new RunnableHelper<Void>();
        ExecutionResult<Void> res = runner.execute(getClass().getClassLoader(), new IExecution<Void>() {
            public Void execute() throws Exception {
                doDeploy(deployable);
                return null;
            }
        });
        res.getResult();
        if (res.hasException()) {
            if (res.getException() instanceof DeployerException) {
                throw (DeployerException) res.getException();
            } else {
                throw new DeployerException("Unable to deploy deployable '" + deployable + "'", res.getException());
            }
        }

        deployedDeployables.put(getURL(deployable), deployable);
    }

    /**
     * {@inheritDoc}
     */
    public void undeploy(final IDeployable<T> deployable) throws DeployerException {
        check(deployable);

        RunnableHelper<Void> runner = new RunnableHelper<Void>();
        ExecutionResult<Void> res = runner.execute(getClass().getClassLoader(), new IExecution<Void>() {
            public Void execute() throws Exception {
                doUndeploy(deployable);
                return null;
            }
        });
        res.getResult();

        if (res.hasException()) {
            if (res.getException() instanceof DeployerException) {
                throw (DeployerException) res.getException();
            } else {
                throw new DeployerException("Unable to undeploy deployable '" + deployable + "'", res.getException());
            }
        }
        deployedDeployables.remove(getURL(deployable));
    }

    /**
     * Undeploy all previously deployed deployables.
     */
    public void stop() {
        for (IDeployable<T> deployable : new ArrayList<IDeployable<T>>(deployedDeployables.values())) {
            try {
                undeploy(deployable);
            } catch (Exception e) {
                logger.error("Cannot undeploy the deployable ''{0}''", deployable, e);
            }
        }
    }

    /**
     * Allows to get the URL from the given deployable.
     * @param deployable the given deployable.
     * @return an URL object of this deployable
     * @throws DeployerException if the URL can't be obtained.
     */
    protected URL getURL(final IDeployable<T> deployable) throws DeployerException {
        // Get URL
        URL url = null;
        try {
            url = deployable.getArchive().getURL();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get URL from deployable '" + deployable + "'.", e);
        }

        // Get URL
        return url;
    }

    /**
     * Allows to get a File from the given Deployable.
     * @param deployable the given deployable.
     * @return a File object of this deployable
     * @throws DeployerException if the File can't be obtained.
     */
    protected File getFile(final IDeployable<T> deployable) throws DeployerException {
        // Get File
        return URLUtils.urlToFile(getURL(deployable));
    }

    /**
     * Checks if the given deployable is deployed or not.
     * @param deployable test if a given deployable is already deployed.
     * @return true if the deployable is deployed.
     * @throws DeployerException if the undeploy operation fails.
     */
    public boolean isDeployed(final IDeployable<T> deployable) throws DeployerException {
        check(deployable);
        return deployedDeployables.containsKey(getURL(deployable));
    }

    /**
     * Returns a map between an URL and the associated deployable that has been deployed.
     * @return A map between an URL and the associated deployable that has been deployed.
     */
    public Map<URL, IDeployable<T>> getDeployedDeployables() {
        return deployedDeployables;
    }

    /**
     * Check if the given deployable is supported.
     * @param deployable the deployable to check.
     * @throws DeployerException if the deployable is not supported.
     */
    protected void check(final IDeployable<?> deployable) throws DeployerException {
        if (!supports(deployable)) {
            throw new DeployerException("The deployment of the deployable'" + deployable
                    + "' is not supported by this deployer.");
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<IDeploymentReport> deploy(final List<IDeployable<T>> deployables)  {
        List<IDeploymentReport> deploymentReports = doDeploy(deployables);
        for (IDeploymentReport deploymentReport: deploymentReports) {
            if (deploymentReport.isDeploymentOk()) {
                IDeployable deployable = deploymentReport.getDeployable();
                try {
                    this.deployedDeployables.put(getURL(deployable), deployable);
                } catch (DeployerException e) {
                    logger.error("Cannot add deployable ''{0}'' to the list of deployed deployables", deployable, e);
                }
            }
        }
        return deploymentReports;
    }

    /**
     * Deploy a list of deployable.
     * @param deployables The list of deployable
     * @return the list of {@link IDeploymentReport}
     */
    public List<IDeploymentReport> doDeploy(final List<IDeployable<T>> deployables) {
        return super.deploy(deployables);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<IDeploymentReport> undeploy(final List<IDeployable<T>> deployables) {
        List<IDeploymentReport> deploymentReports = doUndeploy(deployables);
        for (IDeploymentReport deploymentReport: deploymentReports) {
            if (deploymentReport.isDeploymentOk()) {
                IDeployable deployable = deploymentReport.getDeployable();
                try {
                    this.deployedDeployables.remove(getURL(deployable));
                } catch (DeployerException e) {
                    logger.error("Cannot remove deployable ''{0}'' to the list of deployed deployables", deployable, e);
                }
            }
        }
        return deploymentReports;
    }

    /**
     * Undeploy a list of deployable.
     * @param deployables The list of deployable to undeploy
     * @return the list of {@link IDeploymentReport}
     */
    public List<IDeploymentReport> doUndeploy(final List<IDeployable<T>> deployables) {
        return super.undeploy(deployables);
    }

}
