/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployableMetadataManagerImpl.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.deployable.metadata;

import java.util.HashMap;
import java.util.Map;

import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployable.metadata.DeployableMetadataException;
import org.ow2.util.ee.deploy.api.deployable.metadata.DeployableMetadataManager;
import org.ow2.util.ee.deploy.api.deployable.metadata.IDeployableMetadata;
import org.ow2.util.ee.deploy.api.deployable.metadata.IDeployableMetadataFactory;

/**
 * DefaultDeployableMetadataFactory.
 * @author Gael Lalire
 */
public class DeployableMetadataManagerImpl extends DeployableMetadataManager {

    /**
     * Factory map.
     */
    private Map<Class<?>, Map<Class<?>, IDeployableMetadataFactory<?, ?>>> factoryMap;

    /**
     * Constructor.
     */
    public DeployableMetadataManagerImpl() {
        factoryMap = new HashMap<Class<?>, Map<Class<?>, IDeployableMetadataFactory<?, ?>>>();
    }

    /**
     * Create metadata from a deployable with a registred factory.
     * @param deployable the deployable
     * @param deployableMetadataClass the class of metadata
     * @param <T> the deployable type
     * @param <Z> the metadata type
     * @return new metadata
     * @throws DeployableMetadataException if metadata can not be created
     */
    @SuppressWarnings("unchecked")
    @Override
    public <T extends IDeployable<T>, Z extends IDeployableMetadata<T>> Z createDeployableMetadata(final T deployable,
            final Class<Z> deployableMetadataClass) throws DeployableMetadataException {
        Map<Class<?>, IDeployableMetadataFactory<?, ?>> map = factoryMap.get(deployableMetadataClass);
        IDeployableMetadataFactory<T, Z> factory = null;
        for (Class<?> clazz : deployable.getClass().getInterfaces()) {
            IDeployableMetadataFactory<?, ?> factoryTmp = map.get(clazz);
            if (factoryTmp != null) {
                factory = (IDeployableMetadataFactory<T, Z>) factoryTmp;
                break;
            }
        }
        if (factory == null) {
            throw new DeployableMetadataException("No factory found for the type '" + deployableMetadataClass + "'.");
        }
        return factory.createDeployableMetadata(deployable);
    }

    /**
     * Add a factory.
     * @param clazz the supported deployable interface
     * @param deployableMetadataFactory the metadata factory
     * @param <T> the supported deployable class
     * @param <Z> the deployable metadata class
     * @return true if your factory is assigned to deployable type
     */
    @Override
    public <T extends IDeployable<T>, Z extends IDeployableMetadata<T>> boolean addDeployableMetadataFactory(
            final Class<T> clazz, final IDeployableMetadataFactory<T, Z> deployableMetadataFactory) {
        Class<?> deployableMetadataClass = deployableMetadataFactory.getDeployableMetadataClass();
        Map<Class<?>, IDeployableMetadataFactory<?, ?>> map = factoryMap.get(deployableMetadataClass);
        if (map == null) {
            map = new HashMap<Class<?>, IDeployableMetadataFactory<?, ?>>();
            factoryMap.put(deployableMetadataClass, map);
        }
        if (!map.containsKey(deployableMetadataClass)) {
            map.put(clazz, deployableMetadataFactory);
            return true;
        }
        return false;
    }

    /**
     * Remove a factory.
     * @param clazz the supported deployable interface
     * @param <T> the supported deployable class
     * @param <Z> the deployable metadata class
     * @param deployableMetadataFactory the metadata factory
     */
    @Override
    public <T extends IDeployable<T>, Z extends IDeployableMetadata<T>> void removeDeployableMetadataFactory(
            final Class<T> clazz, final IDeployableMetadataFactory<T, Z> deployableMetadataFactory) {
        Class<?> deployableMetadataClass = deployableMetadataFactory.getDeployableMetadataClass();
        Map<Class<?>, IDeployableMetadataFactory<?, ?>> map = factoryMap.get(deployableMetadataClass);
        if (map == null) {
            return;
        }
        IDeployableMetadataFactory<?, ?> factory = map.get(clazz);
        if (factory.equals(deployableMetadataFactory)) {
            factoryMap.remove(deployableMetadataClass);
        }
    }

}
