/**
 * EasyBeans
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: EARDeployableImpl.java 5928 2011-07-01 10:06:18Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.deployable;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.ee.deploy.api.deployable.CARDeployable;
import org.ow2.util.ee.deploy.api.deployable.EARDeployable;
import org.ow2.util.ee.deploy.api.deployable.EJB21Deployable;
import org.ow2.util.ee.deploy.api.deployable.EJB3Deployable;
import org.ow2.util.ee.deploy.api.deployable.EJBDeployable;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployable.LibDeployable;
import org.ow2.util.ee.deploy.api.deployable.OSGiDeployable;
import org.ow2.util.ee.deploy.api.deployable.RARDeployable;
import org.ow2.util.ee.deploy.api.deployable.UnknownDeployable;
import org.ow2.util.ee.deploy.api.deployable.WARDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployment.ear.ConnectorModule;
import org.ow2.util.ee.deploy.api.deployment.ear.EJBModule;
import org.ow2.util.ee.deploy.api.deployment.ear.IEARInfo;
import org.ow2.util.ee.deploy.api.deployment.ear.JavaModule;
import org.ow2.util.ee.deploy.api.deployment.ear.WebModule;
import org.ow2.util.ee.deploy.impl.deployment.ear.ConnectorModuleImpl;
import org.ow2.util.ee.deploy.impl.deployment.ear.EARInfoImpl;
import org.ow2.util.ee.deploy.impl.deployment.ear.EJBModuleImpl;
import org.ow2.util.ee.deploy.impl.deployment.ear.JavaModuleImpl;
import org.ow2.util.ee.deploy.impl.deployment.ear.WebModuleImpl;
import org.ow2.util.ee.deploy.impl.helper.EarUnpackOpts;
import org.ow2.util.ee.deploy.impl.helper.UnpackDeployableHelper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.xml.DocumentParser;
import org.ow2.util.xml.DocumentParserException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Implementation for an EAR.
 * @author Florent Benoit
 */
public class EARDeployableImpl extends AbsDeployable<EARDeployable> implements EARDeployable {

    /**
     * Application.xml file entry in the EAR file.
     */
    private static final String APPLICATION_XML_ENTRY = "META-INF/application.xml";

    /**
     * Bundle directory of an EAR.
     */
    private static final String BUNDLE_DIRECTORY = "bundle";

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(EARDeployableImpl.class);

    /**
     * Info of the application.xml file of this EAR.
     */
    private EARInfoImpl earInfo = null;

    /**
     * List of all deployables.
     */
    private List<IDeployable<?>> allDeployables = null;

    /**
     * List of EJB 2.1 Deployables.
     */
    private List<EJB21Deployable> ejb21Deployables = null;

    /**
     * List of EJB 3 Deployables.
     */
    private List<EJB3Deployable> ejb3Deployables = null;

    /**
     * List of War Deployables.
     */
    private List<WARDeployable> warDeployables = null;

    /**
     * List of War Deployables.
     */
    private List<RARDeployable> rarDeployables = null;

    /**
     * List of War Deployables.
     */
    private List<CARDeployable> carDeployables = null;

    /**
     * List of Libraries Deployables.
     */
    private List<LibDeployable> libDeployables = null;

    /**
     * List of OSGi Deployables.
     */
    private List<OSGiDeployable> bundleDeployables = null;

    /**
     * List of relative files that are libraries.
     */
    private List<String> classpathLibraries = null;

    /**
     *  Map uri-> deployable.
     */
    private Map<URI, IDeployable<?>> mapUrl = null;

    /**
     * Defines and create a deployable for the given archive.
     * @param archive the given archive.
     */
    public EARDeployableImpl(final IArchive archive) {
        super(archive);
        this.mapUrl = new HashMap<URI, IDeployable<?>>();
        this.allDeployables = new LinkedList<IDeployable<?>>();
        this.ejb21Deployables = new LinkedList<EJB21Deployable>();
        this.ejb3Deployables = new LinkedList<EJB3Deployable>();
        this.warDeployables = new LinkedList<WARDeployable>();
        this.rarDeployables = new LinkedList<RARDeployable>();
        this.carDeployables = new LinkedList<CARDeployable>();
        this.libDeployables = new LinkedList<LibDeployable>();
        this.bundleDeployables = new LinkedList<OSGiDeployable>();

        this.classpathLibraries = new LinkedList<String>();

        // analyze XML DD entry (if any)
        analyzeDD();
    }

    /**
     * Analyze the META-INF/application.xml entry if it is present.
     */
    private void analyzeDD() {
        // Compute URL to the META-INF/application.xml file.
        URL applicationXMLURL = null;
        try {
            applicationXMLURL = getArchive().getResource(APPLICATION_XML_ENTRY);
        } catch (ArchiveException e) {
            throw new IllegalStateException("Cannot get entry '" + APPLICATION_XML_ENTRY + "' on the archive '"
                    + getArchive().getName() + "'.", e);
        }

        // No XML file, just finish the treatment
        if (applicationXMLURL == null) {
            return;
        }

        // Create an XML representation of the file
        this.earInfo = new EARInfoImpl();

        // Get document without validation
        Document document = null;
        try {
            document = DocumentParser.getDocument(applicationXMLURL, false, null);
        } catch (DocumentParserException e) {
            throw new IllegalStateException("Cannot parse the url", e);
        }

        // Root element = <application>
        Element applicationRootElement = document.getDocumentElement();


        // Library directory
        NodeList libraryDirectoryList = applicationRootElement.getElementsByTagName("library-directory");
        // Should be only one element
        if (libraryDirectoryList.getLength() > 0) {
            Element libraryDirectoryElement = (Element) libraryDirectoryList.item(0);
            // Get value
            String nodeValue = "";
            Node txtNode = libraryDirectoryElement.getFirstChild();
            if (txtNode != null) {
                nodeValue = txtNode.getNodeValue().trim();
            }
            earInfo.setLibraryDirectory(nodeValue);
        }

        NodeList moduleList = applicationRootElement.getElementsByTagName("module");
        for (int i = 0; i < moduleList.getLength(); i++) {
            Element moduleElement = (Element) moduleList.item(i);

            // Now, we get the <module> element, need to get connector, ejb, web
            // or java elements

            // connector
            NodeList connectorList = moduleElement.getElementsByTagName("connector");
            NodeList altDDconnList = moduleElement.getElementsByTagName("alt-dd");
            for (int j = 0; j < connectorList.getLength(); j++) {
                Node connNode = connectorList.item(j).getFirstChild();
                ConnectorModuleImpl connectorModule = new ConnectorModuleImpl();
                connectorModule.setPath(connNode.getNodeValue().trim());
                if (altDDconnList.getLength() > 0) {
                    // there is a altDD element for this module
                    Element altddElement = (Element) altDDconnList.item(0);
                    connectorModule.setAltDd(altddElement.getFirstChild().getNodeValue().trim());
                }
                earInfo.addConnectorModule(connectorModule);
            }

            // EJB
            NodeList ejbList = moduleElement.getElementsByTagName("ejb");
            NodeList altDDejbList = moduleElement.getElementsByTagName("alt-dd");
            for (int j = 0; j < ejbList.getLength(); j++) {
                Node ejbNode = ejbList.item(j).getFirstChild();
                EJBModuleImpl ejbModule = new EJBModuleImpl();
                ejbModule.setPath(ejbNode.getNodeValue().trim());
                if (altDDejbList.getLength() > 0) {
                    // there is a altDD element for this module
                    Element altddElement = (Element) altDDejbList.item(0);
                    ejbModule.setAltDd(altddElement.getFirstChild().getNodeValue().trim());
                }
                earInfo.addEJBModule(ejbModule);
            }

            // Java
            NodeList javaList = moduleElement.getElementsByTagName("java");
            NodeList altDDjavaList = moduleElement.getElementsByTagName("alt-dd");
            for (int j = 0; j < javaList.getLength(); j++) {
                Node javaNode = javaList.item(j).getFirstChild();
                JavaModuleImpl javaModule = new JavaModuleImpl();
                javaModule.setPath(javaNode.getNodeValue().trim());
                if (altDDjavaList.getLength() > 0) {
                    // there is a altDD element for this module
                    Element altddElement = (Element) altDDejbList.item(0);
                    javaModule.setAltDd(altddElement.getFirstChild().getNodeValue().trim());
                }
                earInfo.addJavaModule(javaModule);
            }

            // Web
            NodeList webList = moduleElement.getElementsByTagName("web");
            NodeList altDDwebList = moduleElement.getElementsByTagName("alt-dd");
            for (int j = 0; j < webList.getLength(); j++) {
                Element webElement = (Element) webList.item(j);

                // Need to get web-uri and context-root element
                String path = webElement.getElementsByTagName("web-uri").item(0).getFirstChild().getNodeValue().trim();
                String contextRoot = webElement.getElementsByTagName("context-root").item(0).getFirstChild().getNodeValue()
                .trim();

                // Fix invalid context-root (no / at the beginning of the context, except for root context)
                if (contextRoot.startsWith("/") && !contextRoot.equals("/")) {
                    logger.debug("Context-Root ''{0}'' for module ''{1}'' contains invalid starting / in the name. Fixing it.",
                            contextRoot, path);
                    int c = 0;
                    while (contextRoot.charAt(c) == '/') {
                        c++;
                    }
                    contextRoot = contextRoot.substring(c);
                }
                WebModuleImpl webModule = new WebModuleImpl();
                webModule.setPath(path);
                webModule.setContextRoot(contextRoot);

                if (altDDwebList.getLength() > 0) {
                    // there is a altDD element for this module
                    Element altddElement = (Element) altDDejbList.item(0);
                    webModule.setAltDd(altddElement.getFirstChild().getNodeValue().trim());
                }
                earInfo.addWebModule(webModule);

            }
        }
    }

    /**
     * @return the list of the EJB deployables for this EAR.
     */
    public List<EJBDeployable<?>> getEJBDeployables() {
        // Sum of EJB 2.1 and EJB 3
        List<EJBDeployable<?>> ejbDeployables = new LinkedList<EJBDeployable<?>>();

        // 2.1
        for (EJB21Deployable deployable : this.ejb21Deployables) {
            ejbDeployables.add(deployable);
        }

        // 3
        for (EJB3Deployable deployable : this.ejb3Deployables) {
            ejbDeployables.add(deployable);
        }

        return ejbDeployables;

    }

    /**
     * @return the list of the EJB 2.1 deployables for this EAR.
     */
    public List<EJB21Deployable> getEJB21Deployables() {
        return ejb21Deployables;
    }

    /**
     * @return the list of the EJB 3 deployables for this EAR.
     */
    public List<EJB3Deployable> getEJB3Deployables() {
        return ejb3Deployables;
    }

    /**
     * @return the list of the War deployables for this EAR.
     */
    public List<WARDeployable> getWARDeployables() {
        return warDeployables;
    }

    /**
     * @return the list of the Rar deployables for this EAR.
     */
    public List<RARDeployable> getRARDeployables() {
        return rarDeployables;
    }

    /**
     * @return the list of the Car deployables for this EAR.
     */
    public List<CARDeployable> getCARDeployables() {
        return carDeployables;
    }

    /**
     * @return the list of the Library deployables for this EAR.
     */
    public List<LibDeployable> getLibDeployables() {
        return libDeployables;
    }

    /**
     * @return the list of the OSGi deployables for this EAR.
     */
    public List<OSGiDeployable> getOSGiDeployables() {
        return bundleDeployables;
    }

    /**
     * Add the given Deployable to this EAR deployable.
     * @param deployable the given deployable object to add
     */
    public void addDeployable(final IDeployable<?> deployable) {
        addDeployable(deployable, null);
    }

    /**
     * Add the given Deployable to this EAR deployable.
     * @param deployable the given deployable object to add
     * @param earUnpackOpts options to unpack an ear
     */
    public void addDeployable(final IDeployable<?> deployable, final EarUnpackOpts earUnpackOpts) {

        // If there is an XML file, ignore the type that we want to add, use the
        // type specified by the XML file.
        // Also, change the type to a library if it is in the lib/ folder
        IDeployable<?> modifiedDeployable = changeTypeIfSpecified(deployable);

        // Change the type of the given deployable if it is a library for this EAR
        modifiedDeployable = checkLibDeployable(modifiedDeployable);

        // Change type if it is in bundle directory
        modifiedDeployable = updateDeployableIfBundle(modifiedDeployable);

        // Analyze the classpath of this module
        analyzeClassPathModules(modifiedDeployable);

        // Add the deployable in the global list
        allDeployables.add(modifiedDeployable);

        // Add in the right category depending on the interface
        if (EJB21Deployable.class.isInstance(modifiedDeployable)) {
            // EJB 2.1
            ejb21Deployables.add(EJB21Deployable.class.cast(modifiedDeployable));
        } else if (EJB3Deployable.class.isInstance(modifiedDeployable)) {
            // EJB 3
            ejb3Deployables.add(EJB3Deployable.class.cast(modifiedDeployable));
        } else if (WARDeployable.class.isInstance(modifiedDeployable)) {
            // WAR
            WARDeployable warDeployable = WARDeployable.class.cast(modifiedDeployable);
            if (earUnpackOpts != null && earUnpackOpts.isWarAutoUnpacked()) {
                // unpack the war before add it
                try {
                    // Create a new Folder
                    File rootFolder = new File(System.getProperty("java.io.tmpdir") + File.separator
                            + UnpackDeployableHelper.DEFAULT_FOLDER + "-" + System.getProperty("user.name", "default")
                            + File.separator + getShortName());
                    rootFolder.mkdirs();
                    warDeployable = UnpackDeployableHelper.unpack(warDeployable, rootFolder, warDeployable.getShortName());
                } catch (DeployerException e) {
                    logger.error("Unable to unpack a war", e);
                }
            }
            warDeployables.add(warDeployable);
            // Try to set the right context-root for this module
            setContextRoot(warDeployable);
        } else if (RARDeployable.class.isInstance(modifiedDeployable)) {
            // RAR
            rarDeployables.add(RARDeployable.class.cast(modifiedDeployable));
        } else if (CARDeployable.class.isInstance(modifiedDeployable)) {
            // CAR
            carDeployables.add(CARDeployable.class.cast(modifiedDeployable));
        } else if (LibDeployable.class.isInstance(modifiedDeployable)) {
            // Library
            libDeployables.add(LibDeployable.class.cast(modifiedDeployable));
        } else if (OSGiDeployable.class.isInstance(modifiedDeployable)) {
            // Bundle
            bundleDeployables.add(OSGiDeployable.class.cast(modifiedDeployable));
        } else if (UnknownDeployable.class.isInstance(modifiedDeployable)) {
            // Do nothing with this
            logger.debug("Found an unknown deployable {0}", modifiedDeployable);
        } else {
            // Not supported !
            throw new IllegalStateException("The EAR deployable archive doesn't support the include of archive with '"
                    + modifiedDeployable.getClass() + "' class object, deployable ='" + modifiedDeployable + "'.");
        }

        // Add mapping between URL and deployable
        try {
            URL urldeployable = modifiedDeployable.getArchive().getURL();
            mapUrl.put(urldeployable.toURI(), modifiedDeployable);
        } catch (ArchiveException e) {
            throw new IllegalStateException("Cannot get URL of the deployable ", e);
        } catch (URISyntaxException e) {
            throw new IllegalStateException("Cannot get URL of the deployable ", e);
        }
    }

    /**
     * Analyze the classpath of the given deployable.
     * @param deployable the deployable to analyze
     */
    protected void analyzeClassPathModules(final IDeployable<?> deployable) {
        // Read the manifest
        URL urlManifest = null;
        try {
            urlManifest = deployable.getArchive().getResource("META-INF/MANIFEST.MF");
        } catch (ArchiveException e) {
           logger.error("Unable to read the MANIFEST in the deployable {0}", deployable, e);
        }

        // No manifest, stop now
        if (urlManifest == null) {
            logger.debug("No MANIFEST found in the deployable {0}", deployable);
            return;
        }

        // Create Manifest Object from the URL stream
        Manifest manifest = new Manifest();
        URLConnection urlConnection = null;
        try {
            urlConnection = urlManifest.openConnection();
        } catch (IOException e) {
            logger.error("Unable to open connection on URL {0}", urlManifest, e);
            return;
        }

        // Disable cache
        urlConnection.setUseCaches(false);

        InputStream is = null;
        try {
            is = urlConnection.getInputStream();
            // Load stream
            manifest.read(is);
        } catch (IOException e) {
            logger.error("Unable to read stream of {0}", urlManifest, e);
            return;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    logger.error("Unable to close stream of {0}", urlManifest, e);
                    return;
                }
            }
        }

        // A Manifest file has been found, analyze "Class-Path:" entry

        // Get attributes (classpath)
        Attributes attributes = manifest.getMainAttributes();
        String classPath = attributes.getValue(Attributes.Name.CLASS_PATH);

        if (classPath != null) {
            // Add the name of each module to the list
            StringTokenizer stringTokenizer = new StringTokenizer(classPath);
            while (stringTokenizer.hasMoreTokens()) {
                String module = stringTokenizer.nextToken();
                // This name is relative from the current file, needs to convert it to an absolute path
                String relativeName = getRelativeNameOfAModuleFromDeployable(deployable, module);
                // add it if not yet present
                if (relativeName != null && !classpathLibraries.contains(relativeName)) {
                    classpathLibraries.add(relativeName);
                }
            }
        }
    }


    /**
     * If there is an XML file, ignore the type that we want to add, use the
     * type specified by the XML file.
     * @param deployable the module of this EAR
     * @return the new deployable if it is a library module.
     */
    protected IDeployable<?> changeTypeIfSpecified(final IDeployable<?> deployable) {

        // No XML infos, don't change anything
        if (earInfo == null) {
            return deployable;
        }

        // Get name within the EAR
        String relativeModuleName = getRelativeNameFromDeployable(deployable);

        // Try to see if it is a module
        // EJB Module
        for (EJBModule ejbModule : earInfo.getEJBs()) {
            // Found
            if (relativeModuleName.equals(ejbModule.getPath())) {
                // Only check that it is a valid EJB
                if (EJBDeployable.class.isInstance(deployable)) {
                    return deployable;
                } else {
                    throw new IllegalArgumentException("The given deployable '" + deployable
                            + "' is not an EJB but it is flagged as an EJB-JAR in the application.xml file");
                }
            }
        }

        // Web Module
        for (WebModule webModule : earInfo.getWebs()) {
            // Found
            if (relativeModuleName.equals(webModule.getPath())) {
                // Change the type to WAR deployable
                return new WARDeployableImpl(deployable.getArchive());
            }
        }

        // JCA connector ?
        for (ConnectorModule connectorModule : earInfo.getConnectors()) {
            // Found
            if (relativeModuleName.equals(connectorModule.getPath())) {
                // Change the type to RAR deployable
                return new RARDeployableImpl(deployable.getArchive());
            }
        }

        // Client ?
        for (JavaModule javaModule : earInfo.getClients()) {
            // Found
            if (relativeModuleName.equals(javaModule.getPath())) {
                // Change the type to Client deployable
                return new CARDeployableImpl(deployable.getArchive());
            }
        }

        // Not found, then it's an unknown file (and may be later changed to a library)
        return new UnknownDeployableImpl(deployable.getArchive());


    }

    /**
     * If it is a library, change the type of the deployable.
     * @param deployable the module of this EAR
     * @return the new deployable if it is a library module.
     */
    protected IDeployable<?> checkLibDeployable(final IDeployable<?> deployable) {

        // Get name within the EAR
        String relativeModuleName = getRelativeNameFromDeployable(deployable);

        String libraryDirectory = null;
        if (earInfo != null) {
            libraryDirectory = earInfo.getLibraryDirectory();
        } else {
            // default value
            libraryDirectory = IEARInfo.DEFAULT_LIBRARY_FOLDER;
        }

        // No library directory, do nothing
        if ("".equals(libraryDirectory)) {
            return deployable;
        }

        // Append a / at the end of the directory name
        if (!libraryDirectory.endsWith("/")) {
            libraryDirectory += "/";
        }


        // It is a library
        if (relativeModuleName.startsWith(libraryDirectory)) {
            return new LibDeployableImpl(deployable.getArchive());
        }

        return deployable;
    }

    /**
     * If it is a bundle (in the bundle/ directory), change the type of the deployable.
     * @param deployable the module of this EAR
     * @return the updated deployable if it is a bundle module.
     */
    protected IDeployable<?> updateDeployableIfBundle(final IDeployable<?> deployable) {

        // Get name within the EAR
        String relativeModuleName = getRelativeNameFromDeployable(deployable);
        String bundleDirectory = BUNDLE_DIRECTORY;

        // No library directory, do nothing
        if ("".equals(bundleDirectory)) {
            return deployable;
        }

        // Append a / at the end of the directory name
        if (!bundleDirectory.endsWith("/")) {
            bundleDirectory += "/";
        }

        // It is a library
        if (relativeModuleName.startsWith(bundleDirectory)) {
            return new OSGiDeployableImpl(deployable.getArchive());
        }

        return deployable;
    }

    /**
     * Return the relative name of the deployable from the EAR deployable.
     * @param deployable the module of the EAR
     * @return the relative name
     */
    protected String getRelativeNameFromDeployable(final IDeployable<?> deployable) {
        // get URL of the EAR file
        URL earURL = null;
        try {
            earURL = getArchive().getURL();
        } catch (ArchiveException e) {
            throw new IllegalStateException("Cannot get the URL for the EAR file '" + this + "'.", e);
        }

        // get the URL of the deployable
        URL deployableURL = null;
        try {
            IDeployable<?> originalDeployable = deployable.getOriginalDeployable();
            if (originalDeployable != null) {
                deployableURL = originalDeployable.getArchive().getURL();
            } else {
                deployableURL = deployable.getArchive().getURL();
            }
        } catch (ArchiveException e) {
            throw new IllegalStateException("Cannot get the URL for the deployable '" + deployable + "'.", e);
        }

        String relativeName = deployableURL.toExternalForm().substring(earURL.toExternalForm().length());

        // If the relative name ends with a '/', it means that it denotes a directory.
        // Always strips the extra '/' at the end
        if (relativeName.endsWith("/")) {
            relativeName = relativeName.substring(0, relativeName.length() - 1);
        }

        return relativeName;
    }

    /**
     * Return the name of a module with a path from the root of the EAR.
     * @param deployable a module of the EAR
     * @param relativeName a relative name from the given deployable
     * @param relative true for returning a relative path false otherwise
     * @return a  relative or absolute path in the EAR scope
     */
    protected String  computePathForModule(final IDeployable<?> deployable, final String relativeName, final boolean relative) {
        // Get URL of the EAR file
        URL earURL = null;
        try {
            earURL = getArchive().getURL();
        } catch (ArchiveException e) {
            throw new IllegalStateException("Cannot get the URL for the EAR file '" + this + "'.", e);
        }

        // Get the URL of the deployable
        URL deployableURL = null;
        try {
            deployableURL = deployable.getArchive().getURL();
        } catch (ArchiveException e) {
            throw new IllegalStateException("Cannot get the URL for the deployable '" + deployable + "'.", e);
        }

        // Adds the relative name with ../ to exclude the name of the current module
        String newName = deployableURL.getPath() + "/../" + relativeName;

        // Build an URI for this new path and get relative name from the EAR
        URI uriTmp = null;
        try {
            uriTmp = new URI(deployableURL.getProtocol() + ":" + newName);
        } catch (URISyntaxException e) {
            throw new IllegalStateException("Cannot build URI", e);
        }
        // Normalize the URI (to resolve ../ etc)
        uriTmp = uriTmp.normalize();

        // Get path within the EAR
        if (relative) {

            // Check that the module is in the EAR path
            try {
                if (!uriTmp.toString().startsWith(earURL.toURI().toString())) {
                    logger.warn("The deployable ''{0}'' is referencing through the Class-Path: entry ''{1}''"
                            + " but this module is not inside the EAR. Ignore it", deployable, relativeName);
                    return null;
                }
            } catch (URISyntaxException e) {
                throw new IllegalStateException("Cannot normalize URI", e);
            }

            try {
                return uriTmp.toString().substring(earURL.toURI().toString().length());

            } catch (URISyntaxException e) {
                throw new IllegalStateException("Cannot normalize URI", e);
            }
        } else {
            return uriTmp.toString();
        }
    }


    /**
     * Return the name of a module with a path from the root of the EAR.
     * @param deployable a module of the EAR
     * @param relativeName a relative name from the given deployable
     * @return the relative name in the EAR scope
     */
    protected String getRelativeNameOfAModuleFromDeployable(final IDeployable<?> deployable, final String relativeName) {
        return computePathForModule(deployable, relativeName, true);
    }


    /**
     * Set the name of the context-root for the given WAR by checking the EAR
     * information (if any).
     * @param warDeployable the given war to analyze
     */
    protected void setContextRoot(final WARDeployable warDeployable) {

        // Extract the relative name of the war from the EAR
        String relativeWarFileName = getRelativeNameFromDeployable(warDeployable);

        // now, try to find a matching value from the parsed application.xml
        // file
        boolean found = false;
        if (earInfo != null) {
            for (WebModule webModule : earInfo.getWebs()) {
                if (relativeWarFileName.equals(webModule.getPath())) {
                    // Found the correct entry
                    String ctxRoot = webModule.getContextRoot();
                    warDeployable.setContextRoot(ctxRoot);
                    if (logger.isDebugEnabled()) {
                        logger.debug("Context root for War deployable ''{0}'' was found in the EAR "
                                + "{1} file, the value is ''{2}''", warDeployable, APPLICATION_XML_ENTRY, ctxRoot);
                    }
                    found = true;
                    break;
                }
            }
        }

        // Not found, apply a default context root
        if (!found) {

            String ctxRoot = warDeployable.getModuleName();
            warDeployable.setContextRoot(ctxRoot);
            if (logger.isDebugEnabled()) {
                logger.debug("The entry for War deployable ''{0}'' was not found in the EAR "
                        + "application.xml file, using a default context-root value ''{1}''", warDeployable, ctxRoot);
            }

        }

    }

    /**
     * Resolve the ClassPath entries. It means that it puts the unknown modules
     * into Library modules.
     */
    public void resolveClassPath() {
        // No classpath, do nothing
        if (classpathLibraries.isEmpty()) {
            return;
        }



        // In order to avoid concurrent modification, use two lists
        List<IDeployable<?>> currentDeployables = new LinkedList<IDeployable<?>>();
        for (IDeployable<?> deployable : allDeployables) {
            currentDeployables.add(deployable);
        }

        // for each deployable that is a classpath library, change its type.
        for (IDeployable<?> deployable : currentDeployables) {
            // Extract the relative name of the module
            String relativeFileName = getRelativeNameFromDeployable(deployable);
            if (classpathLibraries.contains(relativeFileName)) {
                // if not unknown type (display a warning)
                if (!(UnknownDeployable.class.isInstance(deployable) || LibDeployable.class.isInstance(deployable))) {
                    logger.warn("Deployable {0} is marked as LibraryDeployable in MANIFEST Class-Path: entry "
                            + "but this is a Java EE module.", deployable);
                }
                // If not yet a library, change it
                if (!LibDeployable.class.isInstance(deployable)) {
                    // Remove it as the type will be changed
                    allDeployables.remove(deployable);

                    // Change type to Library
                    LibDeployable libDeployable = new LibDeployableImpl(deployable.getArchive());
                    libDeployables.add(libDeployable);
                    allDeployables.add(libDeployable);
                }

            }

        }
    }
    /**
     * Return the path (string form of url) of a module/file in the EAR.
     * @param deployable a module of the EAR
     * @param relativeName a relative name from the given deployable
     * @return the URL  of the module in EAR
     */
    protected String  getPathOfArelativeModuleFromDeployable(final IDeployable<?> deployable, final String relativeName) {
        return computePathForModule(deployable, relativeName, false);
    }

    /**
     * @return EAR information (of the application.xml file).
     */
    public IEARInfo getEARInfo() {
        return earInfo;
    }


    /**
     * @return the list of all deployables for this EAR.
     */
    public List<IDeployable<?>> getAllDeployables() {
      return allDeployables;
    }


    /**
     * @param url  of the wanted deployable
     * @return deployable corresponding to the url
     */
    public IDeployable<?> getDeployable(final URL url) {
        IDeployable<?> deployable = null;
        URI uri = null;
        try {
            uri = url.toURI();
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid URL", e);
        }
        if (mapUrl.containsKey(uri)) {
            deployable = mapUrl.get(uri);
        }
        return deployable;
    }

    /**
     * @param deployable a module of the EAR
     * @return the url of the alt deployment descriptor
     *   file relative to the application's root or null if altDD element is not present
     *
     */

    public URL getAltDDURL(final IDeployable<?> deployable) {

        // No XML infos -> no altDD
        if (earInfo == null) {
            return null;
        }

        //IDeployable<?> deployable = (IDeployable<?>)mydeployable;
        // Get name within the EAR
        String relativeModuleName = getRelativeNameFromDeployable(deployable);
        String path = null;
        // EJB Module ?
        for (EJBModule ejbModule : earInfo.getEJBs()) {
            path = ejbModule.getAltDd();
            if (!(path == null)) {
                if (relativeModuleName.equals(ejbModule.getPath())) {
                    path = getPathOfArelativeModuleFromDeployable(deployable, ejbModule.getAltDd());
                    return stringToURL(path);
                }
            }
        }
        // Web Module ?
        for (WebModule webModule : earInfo.getWebs()) {
            path = webModule.getAltDd();
            if (!(path == null)) {
                if (relativeModuleName.equals(webModule.getPath())) {
                    path = getPathOfArelativeModuleFromDeployable(deployable, webModule.getAltDd());
                    return stringToURL(path);
                }
            }
        }
        // connector ?
        for (ConnectorModule connectorModule : earInfo.getConnectors()) {
            path = connectorModule.getAltDd();
            if (!(path == null)) {
                if (relativeModuleName.equals(connectorModule.getPath())) {
                    path = getPathOfArelativeModuleFromDeployable(deployable, connectorModule.getAltDd());
                    return stringToURL(path);
                }
            }
        }
        // Client ?
        for (JavaModule javaModule : earInfo.getClients()) {
            path = javaModule.getAltDd();
            if (!(path == null)) {
                if (relativeModuleName.equals(javaModule.getPath())) {
                    path = getPathOfArelativeModuleFromDeployable(deployable, javaModule.getAltDd());
                    return stringToURL(path);
                }
            }
        }
        // unknown
        return null;
    }

    /**
     * Build an URL from a String.
     * @param path the path of an URL.
     * @return URL object
     */
    private URL stringToURL(final String path) {
        URL url = null;
        try {
            url = new URL(path);
            return url;
        } catch (MalformedURLException e) {
           throw new IllegalArgumentException("Invalid path '" + path + "'.", e);
        }
    }


    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("EAR '");
        sb.append(getShortName());
        sb.append("', application.xml ");
        if (earInfo != null) {
            sb.append(" provided");
        } else {
            sb.append(" N/A");
        }
        sb.append(", Content: ");
        sb.append(getDeployablesInfo("EJB-JAR(s)", getEJBDeployables()));
        sb.append(getDeployablesInfo("WAR(s)", getWARDeployables()));
        sb.append(getDeployablesInfo("RAR(s)", getRARDeployables()));
        sb.append(getDeployablesInfo("Libraries", getLibDeployables()));
        sb.append(getDeployablesInfo("OSGi bundle(s)", getOSGiDeployables()));
        sb.append(getDeployablesInfo("CAR(s)", getCARDeployables()));

        return sb.toString();
    }

    /**
     * Provides list of the given deployables.
     * @param <T> the  deployable type
     * @param title the title of the string result
     * @param deployables the list of deployables
     * @return an informative string
     */
    private <T extends IDeployable<?>> String getDeployablesInfo(final String title, final List<T> deployables) {
        StringBuilder sb = new StringBuilder();
        if (deployables.size() == 0) {
            return "";
        }

        sb.append(", ");
        sb.append(deployables.size());
        sb.append(" ");
        sb.append(title);
        sb.append(" {");
        int i = 0;
        for (IDeployable<?> deployable : deployables) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(getRelativeNameFromDeployable(deployable));
            i++;
        }
        sb.append("}");
        return sb.toString();
    }

}


