/**
 * EasyBeans
 * Copyright (C) 2007-2008 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AbsDeployable.java 4673 2009-02-20 15:26:00Z fornacif $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.impl.deployable;

import java.util.HashMap;
import java.util.Map;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployable.IDeployableInfo;

/**
 * Defines the common stuff for the deployable objects.
 * @param <T> the type of the deployable.
 * @author Florent Benoit
 */
public abstract class AbsDeployable<T extends IDeployable<T>> implements IDeployable<T> {

    /**
     * Archive used for this deployable.
     */
    private IArchive archive = null;

    /**
     * Original Deployable (not unpacked).
     */
    private T originalDeployable = null;

    /**
     * Unpacked version of this deployable.
     */
    private T unpackedDeployable = null;

    /**
     * Map of extensions for this deployable.
     */
    private Map<Class<? extends IDeployableInfo>, IDeployableInfo> extensions = null;

    /**
     * Defines and create a deployable for the given archive.
     * @param archive the given archive.
     */
    public AbsDeployable(final IArchive archive) {
        this.archive = archive;
        this.extensions = new HashMap<Class<? extends IDeployableInfo>, IDeployableInfo>();
    }

    /**
     * Gets the Archive of this deployable object.
     * @return the archive for this deployable.
     */
    public IArchive getArchive() {
        return archive;
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.getClass().getSimpleName());
        sb.append("[archive=");
        sb.append(archive.getName());
        sb.append("]");
        return sb.toString();
    }

    /**
     * If the Deployable has been unpacked, return the initial deployable.
     * @return the deployable before it is unpacked.
     */
    public T getOriginalDeployable() {
        return originalDeployable;
    }

    /**
     * If the EAR has been unpacked, return the initial archive.
     * @param originalDeployable the deployable before it is unpacked.
     */
    public void setOriginalDeployable(final T originalDeployable) {
        this.originalDeployable = originalDeployable;
    }

    /**
     * If the Deployable has been unpacked, return the unpacked deployable.
     * @return the unpacked deployable associated to this deployable.
     */
    public T getUnpackedDeployable() {
        return unpackedDeployable;
    }

    /**
     * If the Deployable has been unpacked, sets the unpacked deployable.
     * @param unpackedDeployable the unpacked deployable associated to this deployable.
     */
    public void setUnpackedDeployable(final T unpackedDeployable) {
        this.unpackedDeployable = unpackedDeployable;
    }

    /**
     * Adds a deployable info that can be used by a deployer.
     * @param extension the implementation of the extension to register.
     */
    public void addExtension(final IDeployableInfo extension) {
        extensions.put(extension.getClass(), extension);
    }

    /**
     * Gets the extension for the given class.
     * @param extensionClass the class to used as a key
     * @return the extension else null if not found.
     */
    public IDeployableInfo getExtension(final Class<?  extends IDeployableInfo> extensionClass) {
        return extensions.get(extensionClass);
    }


    /**
     * Gets a shorter name for this deployable.
     * @return a shorter name based on the URL.
     */
    public String getShortName() {
        IDeployable<?> deployable = this;


        // Gets the root deployable
        while (deployable.getOriginalDeployable() != null) {
            deployable = deployable.getOriginalDeployable();
        }

        // Get URL
        String urlExternalForm;
        try {
            urlExternalForm = deployable.getArchive().getURL().toExternalForm();
        } catch (ArchiveException e) {
            throw new IllegalStateException("Cannot get URL of the deployable '" + this + "'.", e);
        }

        // if it ends by a /, remove this last one
        if (urlExternalForm.charAt(urlExternalForm.length() - 1) == '/') {
            urlExternalForm = urlExternalForm.substring(0, urlExternalForm.length() - 1);
        }

        // get string after the last / (all URL have this, no need to test if
        // there is a / character)
        int slashPos = urlExternalForm.lastIndexOf('/');

        // Return the shorter name
        return urlExternalForm.substring(slashPos + 1, urlExternalForm.length());
    }

    /**
     * Gets the module name for this deployable.
     * @return the module name of this deployable
     */
    public String getModuleName() {
        // Module name is based on the short name
        String moduleName = getShortName();

        // Remove extension
        int dotPos = moduleName.lastIndexOf('.');
        if (dotPos != -1) {
            moduleName = moduleName.substring(0, dotPos);
        }

        return moduleName;
    }

    @Override
    public boolean equals(final Object object) {
        if (!(object instanceof IDeployable)) {
            return false;
        }
        IDeployable<T> other = (IDeployable<T>) object;
        try {
            return this.getArchive().getURL().equals(other.getArchive().getURL());
        } catch (ArchiveException e) {
            return super.equals(object);
        }
    }



}
