/**
 * EasyBeans
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: IDeployerManager.java 6232 2012-04-13 12:37:03Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.api.deployer;

import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.report.IDeploymentReport;

import java.net.URL;
import java.util.List;

/**
 * Interface of Deployer Manager.
 * @author Florent BENOIT
 */
public interface IDeployerManager {

    /**
     * Register a new Deployer on this manager instance.
     * @param deployer the deployer to register
     */
    void register(final IDeployer deployer);

    /**
     * Unregister a Deployer from this manager instance.
     * @param deployer the given deployer to unregister.
     */
    void unregister(final IDeployer deployer);

    /**
     * Deploy a given deployable by finding the first deployer that accept
     * this type of deployable.
     * @param deployable the given deployable to deploy.
     * @throws DeployerException if it cannot be undeployed.
     * @throws UnsupportedDeployerException if no deployer is found for the given deployable.
     */
    void deploy(IDeployable<?> deployable) throws DeployerException, UnsupportedDeployerException;

    /**
     * Undeploy a given deployable by finding the first deployer that accept
     * this type of deployable.
     * @param deployable the given deployable to undeploy.
     * @throws DeployerException if it cannot be undeployed.
     * @throws UnsupportedDeployerException if no deployer is found for the given deployable.
     */
    void undeploy(IDeployable<?> deployable) throws DeployerException, UnsupportedDeployerException;


    /**
     * Checks if the given deployable is deployed or not.
     * @param deployable test if a given deployable is already deployed.
     * @return true if the given deployable is deployed.
     * @throws DeployerException if the isDeployed operation fails.
     * @throws UnsupportedDeployerException if no deployer is found for the given deployable.
     */
    boolean isDeployed(IDeployable<?> deployable) throws DeployerException, UnsupportedDeployerException;

    /**
     * Add a new callback instance that will be invoked at each deployment/undeployment phase.
     * @param callback The given callback
     */
    void addCallback(final IDeployerManagerCallback callback);

    /**
     * Remove a callback instance.
     * @param callback The given callback
     */
    void removeCallback(final IDeployerManagerCallback callback);

    /**
     * Add a new callback instance that will be invoked at each deployment/undeployment phase.
     * @param callback The given callback
     */
     void addCallback(final IDeployerManagerReportCallback callback);

    /**
     * Remove a callback instance.
     * @param callback The given callback
     */
    void removeCallback(final IDeployerManagerReportCallback callback);

    /**
     * Deploy a list of multi type deployable by finding the first deployer that accept
     * each type of deployable.
     * @param deployables the list of deployable to deploy.
     * @return the list of {@link IDeploymentReport}
     */
    List<IDeploymentReport> deploy(final List<IDeployable<?>> deployables);

    /**
     * Undeploy a list of multi type deployable by finding the first deployer that accept each type of deployable.
     * @param deployables The list of deployable to undeploy.
     * @return the list of {@link IDeploymentReport}
     */
    List<IDeploymentReport> undeploy(final List<IDeployable<?>> deployables);

    /**
     * @return classnames of each {@link IDeployer}
     */
    List<String> getDeployerClasses();

}

