/**
 * EasyBeans
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: IDeployer.java 6232 2012-04-13 12:37:03Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.api.deployer;

import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.report.IDeploymentReport;

import java.util.List;

/**
 * Interface allowing to deploy/undeploy the {@link IDeployable} objects.
 * @author Florent Benoit
 */
public interface IDeployer<T extends IDeployable<T>> {

    /**
     * Deploy a deployable. It can be an EJB jar, EAR, WAR, etc.
     * @param deployable a given deployable
     * @throws DeployerException if the deployment is not done.
     * @throws UnsupportedDeployerException No deployer found for deployable.
     */
    void deploy(final IDeployable<T> deployable) throws DeployerException, UnsupportedDeployerException;

    /**
     * Undeploy the given deployable. It can be an EJB jar, EAR, WAR, etc.
     * @param deployable a given deployable to undeploy
     * @throws DeployerException if the undeploy operation fails.
     */
    void undeploy(final IDeployable<T> deployable) throws DeployerException;

    /**
     * Checks if the given deployable is deployed or not.
     * @param deployable test if a given deployable is already deployed.
     * @return true if the given deployable is deployed.
     * @throws DeployerException if the isDeployed operation fails.
     */
    boolean isDeployed(final IDeployable<T> deployable) throws DeployerException;


    /**
     * Checks if the given deployable is supported by the Deployer.
     * @param deployable the deployable to be checked
     * @return true if it is supported, else false.
     */
    boolean supports(final IDeployable<?> deployable);

    /**
     * Deploy deployables. It can be an EJB jar, EAR, WAR, etc. List of same type deployable object
     *
     * @param deployables List of deployables to deploy
     * @return the list of {@link IDeploymentReport}
     */
    List<IDeploymentReport> deploy(final List<IDeployable<T>> deployables);

    /**
     * Undeploy the list of deployable. It can be an EJB jar, EAR, WAR, etc.
     * @param deployables List of deployable to undeploy
     * @return the list of {@link IDeploymentReport}
     */
    List<IDeploymentReport> undeploy(final List<IDeployable<T>> deployables);

    /**
     * Checks if this deployable list is supported by the Deployer.
     * @param deployables the list of deployable to be checked
     * @return true if it is supported, else false.
     */
    boolean supports(final List<IDeployable<?>> deployables);
}
