/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployableMetadataManager.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.deploy.api.deployable.metadata;

import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;

/**
 * Deployable metadata factory.
 * @author Gael Lalire
 */
public abstract class DeployableMetadataManager {

    /**
     * The RI class name.
     */
    private static final String REFERENCE_IMPLEMENTATION_CLASS_NAME =
        "org.ow2.util.ee.deploy.impl.deployable.metadata.DeployableMetadataManagerImpl";

    /**
     * The current instance by thread.
     */
    private static final ThreadLocal<DeployableMetadataManager> INSTANCE =
        new InheritableThreadLocal<DeployableMetadataManager>();

    /**
     * @return the latest set DeployableMetadataManager in current thread or the
     *         RI
     * @throws DeployerException if no ArchiveScannerFactory set and the RI can
     *         not be accessed with current thread classloader
     */
    public static DeployableMetadataManager getInstance() throws DeployerException {
        DeployableMetadataManager deployableMetadataManager = INSTANCE.get();
        if (deployableMetadataManager == null) {
            ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
            try {
                Class<? extends DeployableMetadataManager> clazz = classLoader.loadClass(REFERENCE_IMPLEMENTATION_CLASS_NAME)
                        .asSubclass(DeployableMetadataManager.class);
                deployableMetadataManager = clazz.newInstance();
                INSTANCE.set(deployableMetadataManager);
            } catch (Exception e) {
                throw new DeployerException("Unable to create RI", e);
            }
        }
        return deployableMetadataManager;
    }

    /**
     * Set the DeployableMetadataManager to use in current thread.
     * @param deployableMetadataManager the deployableMetadataManager
     */
    public static void setInstance(final DeployableMetadataManager deployableMetadataManager) {
        INSTANCE.set(deployableMetadataManager);
    }

    /**
     * Create metadata from a deployable with a registred factory.
     * @param deployable the deployable
     * @param deployableMetadataClass the class of metadata
     * @param <T> the deployable type
     * @param <Z> the metadata type
     * @return new metadata
     * @throws DeployableMetadataException if metadata can not be created
     */
    public abstract <T extends IDeployable<T>, Z extends IDeployableMetadata<T>> Z createDeployableMetadata(T deployable,
            Class<Z> deployableMetadataClass) throws DeployableMetadataException;

    /**
     * Add a factory.
     * @param clazz the supported deployable interface
     * @param deployableMetadataFactory the metadata factory
     * @param <T> the supported deployable class
     * @param <Z> the deployable metadata class
     * @return true if your factory is assigned to deployable type
     */
    public abstract <T extends IDeployable<T>, Z extends IDeployableMetadata<T>> boolean addDeployableMetadataFactory(
            Class<T> clazz, IDeployableMetadataFactory<T, Z> deployableMetadataFactory);

    /**
     * Remove a factory.
     * @param clazz the supported deployable interface
     * @param <T> the supported deployable class
     * @param <Z> the deployable metadata class
     * @param deployableMetadataFactory the metadata factory
     */
    public abstract <T extends IDeployable<T>, Z extends IDeployableMetadata<T>> void removeDeployableMetadataFactory(
            Class<T> clazz, IDeployableMetadataFactory<T, Z> deployableMetadataFactory);

}
