/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Transaction.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.transaction;
 
import javax.transaction.xa.XAResource;

/**
 * The Transaction interface allows operations to be performed
 * against the transaction in the target Transaction object.
 * A Transaction object is created corresponding to each global
 * transaction creation. The Transaction object can be used for
 * resource enlistment, synchronization registration, transaction
 * completion, and status query operations.
 * @author Guillaume Sauthier
 */
public interface Transaction {

    /**
     * Complete the transaction represented by this Transaction object.
     * @throws RollbackException Thrown to indicate that the transaction
     *         has been rolled back rather than committed.
     * @throws HeuristicMixedException Thrown to indicate that a
     *         heuristic decision was made and that some relevant
     *         updates have been committed while others have been
     *         rolled back.
     * @throws HeuristicRollbackException Thrown to indicate that a
     *         heuristic decision was made and that all relevant updates
     *         have been rolled back.
     * @throws SecurityException Thrown to indicate that the thread is
     *         not allowed to commit the transaction.
     * @throws IllegalStateException Thrown if the transaction in the
     *         target object is inactive.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void commit() throws RollbackException,
                         HeuristicMixedException,
                         HeuristicRollbackException,
                         SecurityException,
                         IllegalStateException,
                         SystemException;

    /**
     * Disassociate the resource specified from the transaction associated
     * with the target Transaction object.
     * @param xaRes The XAResource object associated with the resource (connection).
     * @param flag One of the values of TMSUCCESS, TMSUSPEND, or TMFAIL.
     * @return <code>true</code> if the resource was delisted successfully;
     *         otherwise <code>false</code>.
     * @throws IllegalStateException Thrown if the transaction in the target object is inactive.
     * @throws SystemException Thrown if the transaction manager encounters an unexpected error condition.
     */
    boolean delistResource(XAResource xaRes,
                           int flag) throws IllegalStateException,
                                            SystemException;
    
    /**
     * Enlist the resource specified with the transaction associated with the
     * target Transaction object.
     * @param xaRes The XAResource object associated with the resource (connection).
     * @return <code>true</code> if the resource was enlisted successfully;
     *         otherwise <code>false</code>.
     * @throws RollbackException Thrown to indicate that the transaction has
     *         been marked for rollback only.
     * @throws IllegalStateException Thrown if the transaction in the target
     *         object is in the prepared state or the transaction is inactive.
     * @throws SystemException Thrown if the transaction manager encounters an
     *         unexpected error condition.
     */
    boolean enlistResource(XAResource xaRes) throws RollbackException,
                                                    IllegalStateException,
                                                    SystemException;
    
    /**
     * Obtain the status of the transaction associated with the target
     * Transaction object.
     * @return The transaction status. If no transaction is associated with the
     * target object, this method returns the Status.NoTransaction value.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    int getStatus() throws SystemException;
    
    /**
     * Register a synchronization object for the transaction currently
     * associated with the target object. The transction manager invokes the
     * beforeCompletion method prior to starting the two-phase transaction
     * commit process. After the transaction is completed, the transaction
     * manager invokes the afterCompletion method.
     * @param sync The {@link Synchronization} object for the transaction
     *        associated with the target object.
     * @throws RollbackException Thrown to indicate that the transaction has
     *         been marked for rollback only.
     * @throws IllegalStateException Thrown if the transaction in the target
     *         object is in the prepared state or the transaction is inactive.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void registerSynchronization(Synchronization sync) throws RollbackException,
                                                              IllegalStateException,
                                                              SystemException;
    
    /**
     * Rollback the transaction represented by this {@link Transaction} object.
     * @throws IllegalStateException Thrown if the transaction in the target
     *         object is in the prepared state or the transaction is inactive.
     * @throws SystemException Thrown if the transaction manager encounters an
     *         unexpected error condition.
     */
    void rollback() throws IllegalStateException, SystemException;
    
    /**
     * Modify the transaction associated with the target object such that the
     * only possible outcome of the transaction is to roll back the transaction.
     * @throws IllegalStateException Thrown if the target object is not associated
     *         with any transaction.
     * @throws SystemException Thrown if the transaction manager encounters an
     *         unexpected error condition.
     */
    void setRollbackOnly() throws IllegalStateException, SystemException;
}
