/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ProviderUtil.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence.spi;


/**
 * The ProviderUtil interface is invoked by the PersistenceUtil implementation to determine the load status of an entity or entity
 * attribute. It is not intended to be invoked by the application.
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 2.0 version.
 */
public interface ProviderUtil {

    /**
     * If the provider determines that the entity has been provided by itself and that the state of the specified attribute has
     * been loaded, this method returns LoadState.LOADED. If the provider determines that the entity has been provided by itself
     * and that either entity attributes with FetchType EAGER have not been loaded or that the state of the specified attribute
     * has not been loaded, this methods returns LoadState.NOT_LOADED. If a provider cannot determine the load state, this method
     * returns LoadState.UNKNOWN. The provider's implementation of this method must not obtain a reference to an attribute value,
     * as this could trigger the loading of entity state if the entity has been provided by a different provider.
     * @param entity
     * @param attributeName name of attribute whose load status is to be determined
     * @return load status of the attribute
     */
    public LoadState isLoadedWithoutReference(Object entity, String attributeName);

    /**
     * If the provider determines that the entity has been provided by itself and that the state of the specified attribute has
     * been loaded, this method returns LoadState.LOADED. If a provider determines that the entity has been provided by itself and
     * that either the entity attributes with FetchType EAGER have not been loaded or that the state of the specified attribute
     * has not been loaded, this method returns return LoadState.NOT_LOADED. If the provider cannot determine the load state, this
     * method returns LoadState.UNKNOWN. The provider's implementation of this method is permitted to obtain a reference to the
     * attribute value. (This access is safe because providers which might trigger the loading of the attribute state will have
     * already been determined by isLoadedWithoutReference. )
     * 
     * @param entity
     * @param attributeName name of attribute whose load status is to be determined
     * @return load status of the attribute
     */
    public LoadState isLoadedWithReference(Object entity, String attributeName);

    /**
     * If the provider determines that the entity has been provided by itself and that the state of all attributes for which
     * FetchType EAGER has been specified have been loaded, this method returns LoadState.LOADED. If the provider determines that
     * the entity has been provided by itself and that not all attributes with FetchType EAGER have been loaded, this method
     * returns LoadState.NOT_LOADED. If the provider cannot determine if the entity has been provided by itself, this method
     * returns LoadState.UNKNOWN. The provider's implementation of this method must not obtain a reference to any attribute value,
     * as this could trigger the loading of entity state if the entity has been provided by a different provider.
     * @param entity whose loaded status is to be determined
     * @return load status of the entity
     */
    public LoadState isLoaded(Object entity);
}
