/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: EntityTransaction.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

/**
 * The EntityTransaction interface is used to control resource transactions on
 * resource-local entity managers. The EntityManager.getTransaction() method
 * returns the EntityTransaction interface.<br>
 * When a resource-local entity manager is used, and the persistence provider
 * runtime throws an exception defined to cause transaction rollback, it must
 * mark the transaction for rollback.
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 1.0 version.
 */
public interface EntityTransaction {

    /**
     * Start a resource transaction.
     * @throws IllegalStateException if isActive() is true.
     */
    void begin();

    /**
     * Commit the current transaction, writing any unflushed changes to the
     * database.
     * @throws IllegalStateException if isActive() is false.
     * @throws RollbackException if the commit fails.
     */
    void commit();

    /**
     * Roll back the current transaction.
     * @throws IllegalStateException if isActive() is false.
     * @throws PersistenceException if an unexpected error condition is
     *         encountered.
     */
    void rollback();

    /**
     * Mark the current transaction so that the only possible outcome of the
     * transaction is for the transaction to be rolled back.
     * @throws IllegalStateException if isActive() is false.
     */
    void setRollbackOnly();

    /**
     * Determine whether the current transaction has been marked for rollback.
     * @return true if the current transaction has been marked for rollback.
     * @throws IllegalStateException if isActive() is false.
     */
    boolean getRollbackOnly();

    /**
     * Indicate whether a transaction is in progress.
     * @return true if a transaction is in progress.
     * @throws PersistenceException if an unexpected error condition is
     *         encountered.
     */
    boolean isActive();
}
