/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Column.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * The Column annotation is used to specify a mapped column for a persistent
 * property or field.
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 1.0 version.
 */
@Target({ METHOD, FIELD })
@Retention(RUNTIME)
public @interface Column {

    /**
     * (Optional) The name of the column.
     */
    String name() default "";

    /**
     * (Optional) Whether the property is a unique key. This is a shortcut for
     * the UniqueConstraint annotation at the table level and is useful for when
     * the unique key constraint is only a single field. This constraint applies
     * in addition to any constraint entailed by primary key mapping and to
     * constraints specified at the table level.
     */
    boolean unique() default false;

    /**
     * (Optional) Whether the database column is nullable.
     */
    boolean nullable() default true;

    /**
     * (Optional) Whether the column is included in SQL INSERT statements
     * generated by the persistence provider.
     */
    boolean insertable() default true;

    /**
     * (Optional) Whether the column is included in SQL UPDATE statements
     * generated by the persistence provider.
     */
    boolean updatable() default true;

    /**
     * (Optional) The SQL fragment that is used when generating the DDL for the
     * column.
     */
    String columnDefinition() default "";

    /**
     * (Optional) The name of the table that contains the column. If absent the
     * column is assumed to be in the primary table.
     */
    String table() default "";

    /**
     * (Optional) The column length. (Applies only if a string-valued column is
     * used.)
     */
    int length() default 255;

    /**
     * (Optional) The precision for a decimal (exact numeric) column. (Applies
     * only if a decimal column is used.)
     */
    int precision() default 0;

    /**
     * (Optional) The scale for a decimal (exact numeric) column. (Applies only
     * if a decimal column is used.)
     */
    int scale() default 0;

}
