/**
 * EasyBeans
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: OneToOne.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import static javax.persistence.FetchType.EAGER;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * The OneToOne annotation defines a single-valued association to another entity
 * that has one-to-one multiplicity. It is not normally necessary to specify the
 * associated target entity explicitly since it can usually be inferred from the
 * type of the object being referenced.
 * @see <a href="http://www.jcp.org/en/jsr/detail?id=220">EJB 3.0 specification</a>
 * @author Florent Benoit
 * @since EJB 3.0 version.
 */
@Target({ METHOD, FIELD })
@Retention(RUNTIME)
public @interface OneToOne {

    /**
     * (Optional) The entity class that is the target of the association.
     */
    Class targetEntity() default void.class;

    /**
     * (Optional) The operations that must be cascaded to the target of the
     * association.
     */
    CascadeType[] cascade() default {};

    /**
     * (Optional) Whether the association should be lazily loaded or must be
     * eagerly fetched. The EAGER strategy is a requirement on the persistence
     * provider runtime that the associated entity must be eagerly fetched. The
     * LAZY strategy is a hint to the persistence provider runtime.
     */
    FetchType fetch() default EAGER;

    /**
     * (Optional) Whether the association is optional. If set to false then a
     * non-null relationship must always exist.
     */
    boolean optional() default true;

    /**
     * (Optional) The field that owns the relationship. The mappedBy element is
     * only specified on the inverse (non-owning) side of the association.
     */
    String mappedBy() default "";
}
