/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.wrapper.printwriter;

import java.io.PrintWriter;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Loggable;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.LoggerFactory;

/**
 * This class is a PrintWriter wrapper. It exports the PrintWriter methods but
 * fowards the message to a Logger. This implementation bufferizes the data when
 * a print method is used. The buffer and the data are always written when a
 * println method is used. No end of line are inserted by the println methods.
 * A line is equals to a monolog message.
 *
 * @author Sebastien Chassande-Barrioz
 */
public class PrintWriterImpl
        extends PrintWriter
        implements Loggable {

	//private static Writer empty = new EmptyWriter();

	/**
	 * The inner logger instance
	 */
	protected Logger logger = null;
	protected LoggerFactory loggerFactory = null;

    protected int level;

	/**
	 * This field is the buffer which represents the current line.
	 */
	protected String currentLine = "";

	/**
	 * This field indicates the setError method was called.
	 */
	protected boolean errors = false;

	/**
	 * It builds a PrintWriterImpl instance. The default level is DEBUG
	 *
	 * @param l is the logger toward which the message must be send
	 * @throws NullPointerException if the parameter is null.
	 */
	public PrintWriterImpl(Logger l) throws NullPointerException {
		super(new EmptyWriter());
		if (l==null)
			throw new NullPointerException("Logger parameter is null");
		logger = l;
        level =  BasicLevel.DEBUG;
	}

    /**
     * It builds a PrintWriterImpl instance. The default level is DEBUG
     *
     * @param logger is the logger toward which the message must be send
     * @param loggerFactory is the loggerFactory of the logger
     * @throws NullPointerException if one of the parameters is null.
     */
    public PrintWriterImpl(Logger logger,
                           LoggerFactory loggerFactory) throws NullPointerException {
        this(logger);
        if (loggerFactory==null)
            throw new NullPointerException("LoggerFactory parameter is null");
        this.loggerFactory = loggerFactory;
    }

	/**
	 * It builds a PrintWriterImpl instance.
	 *
	 * @param l is the logger toward which the message must be send
     * @param level is the level used to log message. 
	 * @throws NullPointerException if the parameter is null.
	 */
	public PrintWriterImpl(Logger l, int level) throws NullPointerException {
		super(new EmptyWriter());
		if (l==null)
			throw new NullPointerException("Logger parameter is null");
		logger = l;
        this.level = level;
	}

    public int getLevel() {
        return level;
    }

    public void setLevel(int level) {
        this.level = level;
    }

    // IMPLEMENTATION OF THE Loggable INTERFACE //
    //------------------------------------------//

    /**
     * Retrieves the logger instance used
     */
    public Logger getLogger() {
        return logger;
    }

    /**
     * Assigns the logger instance to use
     */
    public void setLogger(Logger logger) {
        this.logger = logger;
    }

    /**
     * Retrieves the logger factory instance used
     */
    public LoggerFactory getLoggerFactory() {
        return loggerFactory;
    }

    /**
     * Assigns the logger factory instance to use
     */
    public void setLoggerFactory(LoggerFactory lf) {
        this.loggerFactory = lf;
    }

    // IMPLEMENTATION OF THE PrintWriter CLASS //
    //-----------------------------------------//

	/**
	 * Flush the stream and check its error state.
	 */
	public boolean checkError() {
		if (currentLine.length()>0) {
			logger.log(level, currentLine);
			currentLine = "";
		}
		return errors;
	}
	/**
	 * It writes the buffer if it is not empty
	 */
	public void close() {
		if (currentLine.length()>0) {
			logger.log(level, currentLine);
			currentLine = "";
		}
	}

	/**
	 * It writes the buffer if it is not empty
	 */
	public void flush() {
		if (currentLine.length()>0) {
			logger.log(level, currentLine);
			currentLine = "";
		}
	}

	/**
	 * Print a boolean value in the buffer.
	 */
	public void print(boolean x) {
		currentLine += x;
	}

	/**
	 * Print a character in the buffer.
	 */
	public void print(char x) {
		currentLine += x;
	}

	/**
	 * Print an array of characters in the buffer.
	 */
	public void print(char[] x) {
		currentLine += new String(x);
	}

	/**
	 * Print a double-precision floating-point number in the buffer.
	 */
	public void print(double x) {
		currentLine += x;
	}

	/**
	 * Print a floating-point number in the buffer.
	 */
	public void print(float x) {
		currentLine += x;
	}

	/**
	 * Print an integer in the buffer.
	 */
	public void print(int x) {
		currentLine += x;
	}

	/**
	 * Print a long integer in the buffer.
	 */
	public void print(long x) {
		currentLine += x;
	}

	/**
	 * Print an object in the buffer.
	 */
	public void print(Object x) {
		currentLine += x;
	}

	/**
	 * Print a string in the buffer.
	 */
	public void print(String x) {
		currentLine += x;
	}

	/**
	 * Send the buffer to the logger
	 */
	public void println() {
		logger.log(level, currentLine);
		currentLine = "";
	}

	/**
	 * Send the buffer and a boolean value to the logger
	 */
	public void println(boolean x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}

	/**
	 * Send the buffer and a character to the logger
	 */
	public void println(char x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}

	/**
	 * Send the buffer and an array of characters to the logger
	 */
	public void println(char[] x) {
		logger.log(level, currentLine + new String(x));
		currentLine = "";
	}
	/**
	 * Send the buffer and a a double-precision floating-point number to the
	 * logger.
	 */
	public void println(double x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}

	/**
	 * Send the buffer and a floating-point number to the logger
	 */
	public void println(float x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}

	/**
	 * Send the buffer and an integer to the logger
	 */
	public void println(int x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}


	/**
	 * Send the buffer and a long integer number to the logger
	 */
	public void println(long x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}

	/**
	 * Send the buffer and an object to the logger
	 */
	public void println(Object x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}

	/**
	 * Send the buffer and a String to the logger
	 */
	public void println(String x) {
		logger.log(level, currentLine + x);
		currentLine = "";
	}

	/**
	 * Indicate that an error has occurred.
	 */
	protected void setError() {
		errors = true;
		logger.log(BasicLevel.ERROR, currentLine + "PrintWriter error");
		currentLine = "";
	}

	/**
	 * Write an array of characters in the buffer.
	 */
	public void write(char[] buf) {
		currentLine += new String(buf);
	}

	/**
	 * Write a portion of an array of characters  in the buffer.
	 */
	public void write(char[] buf, int off, int len) {
		currentLine += new String(buf, off, len);
	}

	/**
	 * Write a single character in the buffer.
	 */
	public void write(int c) {
		currentLine += c;
	}

	/**
	 * Write a string in the buffer.
	 */
	public void write(String s) {
		currentLine += s;
	}

	/**
	 * Write a portion of a string in the buffer.
	 */
	public void  write(String s, int off, int len) {
		currentLine += (s!=null ? s.substring(off, len) : "");
	}

}
