/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.common ;

import java.io.Serializable;
import java.util.StringTokenizer;

import org.objectweb.util.monolog.api.Level;
import org.objectweb.util.monolog.api.LevelFactory;

/**
 * This class is the basic implementation of the Level interface. It proviedes
 * also a static method 'evaluate' which permits to obtain the integer value
 * of a level expression.
 *
 * @author Sebastien Chassande-Barrioz
 */
public class LevelImpl implements Level, Serializable {
	int value = 0;
	String name = null;;
	String stringValue = null;

	public LevelImpl(String n, int val) {
		value = val;
		name = n;
		stringValue = String.valueOf(val);
	}

	public LevelImpl(String n, String val, LevelFactory lf) {
		//System.out.println("LevelImpl("+n+", "+val+", lf)");
		stringValue = val;
		value = evaluate(val, lf);
		name = n;
	}

	/**
	 * It retrieves the integer value of the level.
	 */
	public int hashCode() {
		return value;
	}

	/**
	 * It retrieves the string expression of the level. ex: 'DEBUG + 1'
	 */
	public String getStringValue() {
		return stringValue;
	}

	/**
	 * It analyzes a string expression to obtain its integer value. The allowed
	 * expression type are the following:
	 * <ul>
	 * <li>an integer value</li>
	 * <li>another level name</li>
	 * <li>levelName + integerValue</li>
	 * <li>levelName - integerValue</li>
	 * </ul>
	 * @param expr is the string expression which must be evaluated.
	 * @param lf is the LevelFactory which permits to obtain the referenced level.
	 * @return an integer value or 0 if it is impossible to evaluate the
	 * expression.
	 */
	public static int evaluate(String expr, LevelFactory lf) {
		int firstOperande = 0;
		int secondOperande = 0;
		byte operator = 0; // 1=>+  2=>-
		for (
			StringTokenizer st = new StringTokenizer(expr, " +-", true);
			st.hasMoreTokens();) {

			String elem = st.nextToken();
			if (Character.isDigit(elem.charAt(0))) {
				try {
					secondOperande = Integer.parseInt(elem);
				} catch (NumberFormatException e) {
					continue;
				}
			} else if (Character.isLetter(elem.charAt(0))) {
				// The token is another intermediate level
				Level l = lf.getLevel(elem);
				if (l == null) {
                    l = lf.getLevel(elem.toUpperCase());
                    if (l == null) {
                        return 0;
                    }
				}
				firstOperande = l.getIntValue();
			} else if (elem.charAt(0) == '+') {
				operator = 1;
			} else if (elem.charAt(0) == '-') {
				operator = 2;
			} else if (Character.isSpaceChar(elem.charAt(0))) {
				continue;
			}
		}
		if (firstOperande != 0) {
			if (secondOperande != 0 && operator != 0) {
				switch (operator) {
				case 1:
					return firstOperande + secondOperande;
				case 2:
					return firstOperande - secondOperande;
				}
			}
			else {
				return firstOperande;
			}
		}
		else if (secondOperande != 0) {
			return secondOperande;
		}
		return 0;
	}

	public String toString() {
		return "(name=" + name + ", val=" + value + ", sval=" + stringValue + ")";
	}

	// IMPLEMENTATION OF THE Level INTERFACE //
	//---------------------------------------//

	public boolean isComparableWith(Level o) {
		return o instanceof LevelImpl;
	}
	
	public int compareTo(Level o) {
		return value - o.getIntValue();
	}

	public int getIntValue() { 
		return value;
	}

	public String getName() {
		return name;
	}

	public void setName(String n) {
		name = n;
	}
}
