/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.file.monolog;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Handler;
import org.objectweb.util.monolog.api.HandlerFactory;
import org.objectweb.util.monolog.api.Level;
import org.objectweb.util.monolog.api.LevelFactory;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.LoggerFactory;
import org.objectweb.util.monolog.api.MonologFactory;
import org.objectweb.util.monolog.api.TopicalLogger;
import org.objectweb.util.monolog.file.DottedStringTools;
import org.objectweb.util.monolog.wrapper.common.LevelImpl;

/**
 * This class permits to load and store a monolog configuration. The chooseen
 * format is java.util.Properties. It also easy to write or load a Properties
 * from a file. The encoding format is the following:<br/>
 * handler.<handler name>.<property name>=<property value><br/>
 *<br/>
 * level.<level name>=<integer value or string expression><br/>
 *<br/>
 * logger.<dotted logger name>.level=<level name><br/>
 * logger.<dotted logger name>.additivity=<true | false><br/>
 * logger.<dotted logger name>.handler.<number>=<handler name><br/>
 * logger.<dotted logger name>.topic.<number>=<additionnal logger name><br/>
 *<br/>
 * @author Sebastien Chassande-Barrioz
 */
public class PropertiesConfAccess implements Serializable {

    /**
     * 
     */
    private static final long serialVersionUID = -321110630195680214L;
    
    public final static String LOGGER_FIELD = "logger";
    public final static String ACTIVATION = "activation";
    public final static String ADDITIVITY_FIELD = "additivity";
    public final static String USE_PARENT_FIELD = "useParent";
    public final static String HANDLER_FIELD = "handler";
    public final static String LEVEL_FIELD = "level";
    public final static String TOPIC_FIELD = "topic";
    public final static String CLEAN_HANDLERS_FIELD = "cleanHandlers";
    public final static char DOT = '.';

    public final static String HANDLER_TYPE_ATTRIBUTE = "type";

    public final static String HANDLER_TYPE_ATTRIBUTE_FILE_VALUE = "file";
    public final static String HANDLER_TYPE_ATTRIBUTE_CONSOLE_VALUE = "console";
    public final static String HANDLER_TYPE_ATTRIBUTE_ROLLING_FILE_VALUE = "rollingfile";
    public final static String HANDLER_TYPE_ATTRIBUTE_NTEVENT_VALUE = "ntevent";
    public final static String HANDLER_TYPE_ATTRIBUTE_JMX_VALUE = "jmx";

    public static boolean debug = new Boolean(
			System.getProperty("monolog.debug")).booleanValue();
    
    /**
     * Map which associates an handler to replace with a logger.
     */
    private Map handlersToReplace = new HashMap();

    public static void load(Properties prop, LoggerFactory lof, HandlerFactory hf,
                            LevelFactory lef) throws Exception {
        new PropertiesConfAccess().read(prop, lof, hf, lef);
    }

    public static void load(Properties prop, MonologFactory mf) throws Exception {
        new PropertiesConfAccess().read(prop, mf, mf, mf);
    }

    public static void store(Properties prop, LoggerFactory lof, HandlerFactory hf,
                             LevelFactory lef) throws Exception {
        new PropertiesConfAccess().write(prop, lof, hf, lef);
    }

    public static void store(Properties prop, MonologFactory mf) throws Exception {
        new PropertiesConfAccess().write(prop, mf, mf, mf);
    }

    public void read(Properties prop, LoggerFactory lof, HandlerFactory hf,
                     LevelFactory lef) throws Exception {
	    read(prop, (MonologFactory) lof);
    }
    public void read(Properties prop, MonologFactory mf) throws Exception {
        //Remove the handlers of the loggers which have the property
        //CLEAN_HANDLERS_FIELD
        for (Enumeration e = prop.keys(); e.hasMoreElements();) {
            String key = ((String) e.nextElement());
            if (key.startsWith(LOGGER_FIELD + DOT)
                    && key.endsWith(DOT + CLEAN_HANDLERS_FIELD)
                    && Boolean.getBoolean(prop.getProperty(key).trim())) {
                String loggerName = key.substring(
                        LOGGER_FIELD.length() + 1,
                        key.length() - 1 - CLEAN_HANDLERS_FIELD.length());
                ((TopicalLogger) mf.getLogger(loggerName)).removeAllHandlers();
            }
        }
        //Parse logger and level definition
        for (Enumeration en = prop.keys(); en.hasMoreElements();) {
            String key = (String) en.nextElement();
            if (key.startsWith(LOGGER_FIELD + DOT)) {
                parseLoggerProp(prop, key, mf);
            } else if (key.startsWith(LEVEL_FIELD + DOT)) {
                parseLevelProp(prop, key, mf);
            }
        }
        //Parse all handlers 
        Vector hs = new Vector(); //contains the list of handler to activate
        for (Enumeration en = prop.keys(); en.hasMoreElements();) {
            String key = (String) en.nextElement();
            if (key.startsWith(HANDLER_FIELD + DOT)) {
                String handlerName = DottedStringTools.getBegin(
                        DottedStringTools.getEnd(key));
                Handler h = mf.getHandler(handlerName);
                if (h != null && !hs.contains(h)) {
                    //Activate only used handler
                    hs.addElement(h);
                } // do not activate unsued handler
                parseHandlerProp(prop, key, mf);
            }
        }
        //Activates and create used handlers only
        for (int i = 0; i < hs.size(); i++) {
            ((Handler) hs.elementAt(i)).setAttribute("activation", mf);
            Handler h = (Handler) hs.elementAt(i);
            TopicalLogger logger = (TopicalLogger) handlersToReplace.get(h);
            if (logger != null) {
                cleanOldHandler(logger);
                logger.addHandler(h);
                if (debug) {
                    debug("add handler " + h.getName() + " to the logger " + logger.getName());
                }
            } 
        }
    }

    public void write(Properties prop, LoggerFactory lof, HandlerFactory hf,
                      LevelFactory lef) throws Exception {

        // Set the level definitions
        String key = null;
        String val = null;
        Level[] levels = lef.getLevels();
        for (int i = 0; i < levels.length; i++) {
            if (!isDefaultLevel(levels[i])) {
                key = LEVEL_FIELD + DOT + levels[i].getName();
                val = ((LevelImpl) levels[i]).getStringValue();
                debug(key + " " + val);
                prop.put(key, val.trim());
            }
        }

        // Set the handler definitions
        Handler[] handlers = hf.getHandlers();
        for (int i = 0; i < handlers.length; i++) {
            key = HANDLER_FIELD + DOT + handlers[i].getName() + DOT + HANDLER_TYPE_ATTRIBUTE;
            val = handlers[i].getType();
            debug(key + " " + val);
            prop.put(key, val);
            String[] ats = handlers[i].getAttributeNames();
            for (int j = 0; j < ats.length; j++) {
                key = HANDLER_FIELD + DOT + handlers[i].getName() + DOT + ats[j];
	            Object o = handlers[i].getAttribute(ats[j]);
	            if (o instanceof String) {
                    val = (String) o;
                    debug(key + " " + val);
                    prop.put(key, val.trim());
	            }
            }
        }

        //Write the logger definition
        Logger[] currLoggers = lof.getLoggers();
        TopicalLogger[] topicalLoggers = new TopicalLogger[currLoggers.length];
        if (currLoggers != null) {
            int i = 0;
            for (Logger logger : currLoggers) {
                if (logger instanceof TopicalLogger) {
                    topicalLoggers[i] = (TopicalLogger) currLoggers[i];
                } else {
                    throw new IllegalStateException("Unable to convert logger '" + logger + "' to TopicalLogger");
                }
                i++;
            }
        }
        
        TopicalLogger[] loggers = topicalLoggers;
        final String topicPrefix = lof.getTopicPrefix(); 
        for (int i = 0; i < loggers.length; i++) {
            String[] topics = loggers[i].getTopic();
            if (topics.length == 0) {
                throw new Exception(
                        "Impossible to set the definition of a logger without name");
            }
            String topic = loggers[i].getTopic()[0];
            if (topicPrefix != null && topic.startsWith(topicPrefix)) {
                topic = topic.substring(topicPrefix.length());
            }
            String begin = LOGGER_FIELD + DOT + topic;

            if (!loggers[i].getAdditivity()) {
                key = begin + DOT + ADDITIVITY_FIELD;
                val = "false";
                debug(key + " " + val);
                prop.put(key, val.trim());
            }

            Level level = loggers[i].getCurrentLevel();
            if (level != null && level.getIntValue() != BasicLevel.INHERIT) {
                key = begin + DOT + LEVEL_FIELD;
                val = " " + level.getName();
                debug(key + " " + val);
                prop.put(key, val.trim());
            }

            handlers = loggers[i].getHandler();
            if (handlers != null && handlers.length > 0) {
                StringBuffer sb = new StringBuffer();
                String sep = "";
                for (int j = 0; j < handlers.length; j++) {
                    if (handlers[j].getName() != null) {
                        sb.append(sep);
                        sep = ", ";
                        sb.append(handlers[j].getName());
                    }
                }
                debug(begin + DOT + HANDLER_FIELD + " " + sb.toString());
                prop.put(begin + DOT + HANDLER_FIELD, sb.toString());
            }

            if (topics.length > 1) {
                StringBuffer sb = new StringBuffer();
                String sep = "";
                for (int j = 1; j < topics.length; j++) {
                    sb.append(sep);
                    sep = ", ";
                    String ntopic = topics[j];
                    if (topicPrefix != null && ntopic.startsWith(topicPrefix)) {
                        ntopic = ntopic.substring(topicPrefix.length());
                    }
                    sb.append(ntopic);
                }
                debug(begin + DOT + TOPIC_FIELD + " " + sb.toString());
                prop.put(begin + DOT + TOPIC_FIELD, sb.toString());
            }
        }
    }

    /**
     * It checks if the level parameter is a default monolog level. The default
     * monolog levels are the following:
     * <ul>
     * <li>FATAL</li>
     * <li>ERROR</li>
     * <li>WARN</li>
     * <li>INFO</li>
     * <li>DEBUG</li>
     * </ul>
     */
    protected boolean isDefaultLevel(Level l) {
        return (l.getName().equalsIgnoreCase("DEBUG")
                || l.getName().equalsIgnoreCase("INFO")
                || l.getName().equalsIgnoreCase("WARN")
                || l.getName().equalsIgnoreCase("ERROR")
                || l.getName().equalsIgnoreCase("FATAL")
                || l.getName().equalsIgnoreCase("INHERIT")
                );
    }

    //------------ PARSING METHODS ------------//
    //-----------------------------------------//

	/**
	 * It parses a property entry to build or configure a Logger instance.
	 * @param prop is the property where the entry is reachable.
	 * @param key is the entry key
	 * @param lof is the logger factory to use for building a new instance or
	 * fetching existent loggers.
	 * @param hf is the handler factory to use for building a new instance or
	 * fetching existent handlers.
	 * @param lef is the level factory to use for building a new instance or
	 * fetching existent levels.
	 * @return the level instance which has been built or configured
	 * @throws Exception when a parameter is null or the entry is malformed.
	 */
	protected Logger parseLoggerProp(Properties prop,
	                                 String key,
	                                 LoggerFactory lof,
	                                 HandlerFactory hf,
	                                 LevelFactory lef) throws Exception {
		return parseLoggerProp(prop, key, (MonologFactory) lof);
	}
    /**
     * It parses a property entry to build or configure a Logger instance.
     * @param prop is the property where the entry is reachable.
     * @param key is the entry key
     * @param mf is the monolog factory to use for building a new instance or
     * fetching existent loggers, handlers or levels.
     * @return the level instance which has been built or configured
     * @throws Exception when a parameter is null or the entry is malformed.
     */
    protected Logger parseLoggerProp(Properties prop,
                                     String key,
                                     MonologFactory mf) throws Exception {
        if (prop == null
                || key == null
                || mf == null) {
            throw new Exception("The null parameters are not allowed");
        }

        String temp = DottedStringTools.getFirst(key);
        if (temp == null || !temp.equalsIgnoreCase(LOGGER_FIELD)) {
            throw new Exception(
                    "This key is not a Logger property:" + key);
        }

        TopicalLogger logger = null;
        temp = DottedStringTools.getEnd(key);
        String last = DottedStringTools.getLast(temp);
        if (last.equalsIgnoreCase(LEVEL_FIELD)) {
            // Assigns the level of the logger
            logger = (TopicalLogger)
                    mf.getLogger(DottedStringTools.getBegin(temp));
            String levelName = prop.getProperty(key).trim();
            Level l = mf.getLevel(levelName);
            if (l == null) {
                l = parseLevelProp(prop, LEVEL_FIELD + DOT + levelName, mf);
            }
            if (debug) {
                debug("set level to " + l.getName() + " to the logger " + logger.getName());
            }
            logger.setLevel(l);
        } else if (ADDITIVITY_FIELD.equalsIgnoreCase(last)
                || USE_PARENT_FIELD.equalsIgnoreCase(last)) {
            logger = (TopicalLogger)
                    mf.getLogger(DottedStringTools.getBegin(temp));
            boolean a = Boolean.getBoolean(prop.getProperty(key).trim());
            if (debug) {
                debug("set additivity to " + a + " to the logger " + logger.getName());
            }
            logger.setAdditivity(a);
        } else if (CLEAN_HANDLERS_FIELD.equalsIgnoreCase(last)) {
            //Already managed iin the read method
        } else if (HANDLER_FIELD.equalsIgnoreCase(last)) {
            String value = prop.getProperty(key).trim();
            StringTokenizer st = new StringTokenizer(value, ", ;:", false);
            logger = (TopicalLogger)
					mf.getLogger(DottedStringTools.getBegin(temp));
            cleanOldHandler(logger);
            while(st.hasMoreTokens()) {
                String hn = st.nextToken();
                Handler h = mf.getHandler(hn);
                if (h == null) {
                    h = parseHandlerProp(prop, HANDLER_FIELD + DOT + value
                            + DOT + HANDLER_TYPE_ATTRIBUTE, mf);
                }
                if (debug) {
                    debug("add handler " + h.getName() + " to the logger " + logger.getName());
                }
                logger.addHandler(h);
            }
        } else if (TOPIC_FIELD.equalsIgnoreCase(last)) {
            logger = (TopicalLogger)
            	mf.getLogger(DottedStringTools.getBegin(temp));
            String value = prop.getProperty(key).trim();
            StringTokenizer st = new StringTokenizer(value, ", ;:", false);
            cleanOldHandler(logger);
            while(st.hasMoreTokens()) {
                String topic = st.nextToken();
                if (debug) {
                    debug("add topic " + topic + " to the logger " + logger.getName());
                }
                logger.addTopic(topic);
            }
        } else { //old declaration
            temp = DottedStringTools.getBegin(temp); // remove the number
            last = DottedStringTools.getLast(temp); // attribute name
            temp = DottedStringTools.getBegin(temp); // logger name
            logger = (TopicalLogger) mf.getLogger(temp);
            if (last.equalsIgnoreCase(HANDLER_FIELD)) {
                // Assign an handler to a logger
                String value = prop.getProperty(key).trim();
                Handler h = mf.getHandler(value);
                if (h == null) {
                    h = parseHandlerProp(prop, HANDLER_FIELD + DOT + value
                            + DOT + HANDLER_TYPE_ATTRIBUTE, mf);
                }
                handlersToReplace.put(h, logger);
            } else if (last.equalsIgnoreCase(TOPIC_FIELD)) {
                // Assign a topic to a logger
                String topic = prop.getProperty(key).trim();
                if (debug) {
                    debug("add topic " + topic + " to the logger " + logger.getName());
                }
                logger.addTopic(topic);
            } else {
                throw new Exception(
                        "Unknown definition" + key + " " + prop.getProperty(key));
            }
        }
        return logger;
    }

    /**
     * It parses a property entry to build or configure a Handler instance.
     * @param prop is the property where the entry is reachable.
     * @param key is the entry key
     * @param hf is the handler factory to use for building a new instance or
     * fetching existent handlers.
     * @return the handler instance which has been built or configured
     * @throws Exception when a parameter is null or the entry is malformed.
     */
    protected Handler parseHandlerProp(Properties prop,
                                       String key,
                                       HandlerFactory hf) throws Exception {

        if (prop == null
                || key == null
                || hf == null) {
            throw new Exception("The null parameters are not allowed");
        }
        String temp = DottedStringTools.getFirst(key);
        if (temp == null || !temp.equalsIgnoreCase(HANDLER_FIELD)) {
            throw new Exception(
                    "This key is not a Handler field:" + key);
        }
        temp = DottedStringTools.getEnd(key);
        String attribute = DottedStringTools.getLast(temp);
        String handlerName = DottedStringTools.getBegin(temp);
        Handler hc = hf.getHandler(handlerName);
        if (hc == null) {
            String stringType = prop.getProperty(HANDLER_FIELD + DOT + handlerName
                    + DOT + HANDLER_TYPE_ATTRIBUTE, null);
            if (stringType == null) {
                throw new Exception("Impossible to define the handler "
                        + temp + ": the type is not defined");
            }
            hc = hf.createHandler(handlerName, stringType);
            if (hc == null) {
                throw new Exception(
                        "The HandlerFactory does not create the Handler: name="
                        + handlerName + " / type=" + stringType);
            }
        } 
        if (hc != null && !HANDLER_TYPE_ATTRIBUTE.equalsIgnoreCase(attribute)) {
            String v = prop.getProperty(key).trim();
            if (debug) {
                debug("assign property (" + attribute + ", " + v 
                        + ") to the handler '" + handlerName);
            }
            hc.setAttribute(attribute, v);
        }
	    return hc;
    }

    /**
     * It parses a property entry to build or configure a Level instance.
     * @param prop is the property where the entry is reachable.
     * @param key is the entry key
     * @param lef is the level factory to use for building a new instance or
     * fetching existent levels.
     * @return the level instance which has been built or configured
     * @throws Exception when a circular level definition is found or when a
     * parameter is null or the entry is malformed.
     */
    protected Level parseLevelProp(Properties prop,
                                   String key,
                                   LevelFactory lef) throws Exception {
        return parseLevelProp(prop, key, lef, new Vector());
    }

    /**
     * It parses a property entry to build or configure a Level instance.
     * @param prop is the property where the entry is reachable.
     * @param key is the entry key
     * @param lef is the level factory to use for building a new instance or
     * fetching existent levels.
     * @param currentLevelParse the list of current level name which are
     * searched.
     * @throws Exception when a circular level definition is found or when a
     * parameter is null.
     */
    protected Level parseLevelProp(Properties prop,
                                   String key,
                                   LevelFactory lef,
                                   Vector currentLevelParse) throws Exception {
        if (prop == null
                || key == null
                || lef == null) {
            throw new Exception("The null parameters are not allowed");
        }
        String temp = DottedStringTools.getFirst(key);
        if (temp == null || !temp.equalsIgnoreCase(LEVEL_FIELD))
            throw new Exception("This key is not a level field:" + key);

        temp = DottedStringTools.getEnd(key);
        Level l = lef.getLevel(temp);
        if (l == null) {
            if (!prop.containsKey(key)) {
                throw new Exception("Level definition not found: " + key);
            }
            if (currentLevelParse.contains(temp)) {
                throw new Exception("Circular level definition: " + temp);
            }
            String levelValue = prop.getProperty(key).trim();
            String[] depends = getDependsLevel(levelValue);
            if (depends.length > 0) {
                currentLevelParse.addElement(temp);
                for (int i = 0; i < depends.length; i++) {
                    parseLevelProp(prop, LEVEL_FIELD + DOT + depends[i],
                            lef, currentLevelParse);
                }
                currentLevelParse.removeElement(temp);
            }
            l = lef.defineLevel(temp, levelValue);
        }
        return l;
    }

    /**
     * It retrieves the list of the referenced levels in the String parameter.
     * @param expr is a string which can contain level identifier. The string is
     * an arithmetic expression which contains number, operator and levels
     * identifier.
     * @return the list of the referenced levels.
     */
    protected String[] getDependsLevel(String expr) {
        Vector res = new Vector();
        for (StringTokenizer st = new StringTokenizer(expr, " +-", true);
             st.hasMoreTokens();) {
            String elem = st.nextToken();
            if (Character.isLetter(elem.charAt(0))) {
                res.addElement(elem);
            }
        }
        String[] r = new String[res.size()];
        res.copyInto(r);
        return r;
    }

    protected Vector loggerCleaned = null;

    /**
     * Clear all the handlers associated to a logger which are not defined into
     * this file
     */
    protected void cleanOldHandler(TopicalLogger l) throws Exception {
        if (loggerCleaned == null) {
            loggerCleaned = new Vector();
        }
        if (!loggerCleaned.contains(l)) {
            loggerCleaned.addElement(l);
            l.removeAllHandlers();
        }
    }

    /**
     * This exception encapsulates an other exception.
     */
    class NestedException extends Exception {
        
        /**
         * 
         */
        private static final long serialVersionUID = -1218233947556357710L;
        
        Exception nestedException = null;

        public NestedException() {
            super();
        }

        public NestedException(Exception e) {
            super();
            nestedException = e;
        }

        public NestedException(Exception e, String msg) {
            super(msg);
            nestedException = e;
        }

        public Exception getNestedException() {
            return nestedException;
        }

        public void printStackTrace(PrintStream ps) {
            if (nestedException != null) {
                ps.println("Nested: " + getMessage());
                nestedException.printStackTrace(ps);
            } else {
                super.printStackTrace(ps);
            }
        }

        public void printStackTrace(PrintWriter w) {
            if (nestedException != null) {
                w.println("Nested: " + getMessage());
                nestedException.printStackTrace(w);
            } else {
                super.printStackTrace(w);
            }
        }

        public void printStackTrace() {
            printStackTrace(new PrintStream(System.out));
        }

    }

    public void debug(String m) {
        if (debug) {
            System.out.println(m);
        }
    }
}
