/**
 * Copyright (C) 2001-2004 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.file.monolog;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Properties;

/**
 * Permit to read or write a java.util.Properties from an inputStream.
 * The interest of this implementation with regards to the 
 * properties.load(InputStream) method is to sort the properties and to take
 * in account particularity of logging properties.
 * 
 * @author S.Chassande-Barrioz
 */
public class MonologPropertiesIO {
    
	public static final char[] PROP_VAL_SEPS = {'=', ' '}; 
    public static final String[] keywords = { 
            PropertiesConfAccess.LOGGER_FIELD + PropertiesConfAccess.DOT,
            PropertiesConfAccess.HANDLER_FIELD + PropertiesConfAccess.DOT,
            PropertiesConfAccess.LEVEL_FIELD + PropertiesConfAccess.DOT
    };
	
    public static void writeProperties(Properties props, OutputStream os) throws IOException {
    	BufferedWriter w = new BufferedWriter(new OutputStreamWriter(os));
        //Sort properties
        ArrayList keys = new ArrayList(props.keySet());
        // Order properties by the alphabetic order 
        Collections.sort(keys);
        ArrayList loggers = new ArrayList();
        ArrayList handlers = new ArrayList();
        ArrayList levels = new ArrayList();
        for(int i=0; i<keys.size(); i++) {
            String key = (String) keys.get(i);
            if (key.startsWith(PropertiesConfAccess.LOGGER_FIELD)) {
            	if (key.startsWith(PropertiesConfAccess.LOGGER_FIELD 
            			+ PropertiesConfAccess.DOT 
						+ "root" + PropertiesConfAccess.DOT)) {
            		loggers.add(0,key);
            	} else {
            		loggers.add(key);
            	}
            } else if (key.startsWith(PropertiesConfAccess.HANDLER_FIELD)) {
            	handlers.add(key);
            } else if (key.startsWith(PropertiesConfAccess.LEVEL_FIELD)) {
            	levels.add(key);
            }
        }
        keys = null;
        
    	w.write("#Generated by Monolog the ");
    	w.write(DateFormat.getInstance().format(
    			Calendar.getInstance().getTime()));
    	
    	if (levels.size() > 0) {
        	w.newLine();
        	w.newLine();
        	w.write("#Level configuration");
        	w.newLine();
            for(int i=0; i<levels.size();i++) {
                String key = (String) levels.get(i);
            	w.write(key);
            	w.write(PROP_VAL_SEPS[0]);
            	w.write(props.getProperty(key));
            	w.newLine();
            }
    	}

    	if (handlers.size() > 0) {
        	w.newLine();
        	w.newLine();
        	w.write("#Handler configuration");
        	w.newLine();
            for(int i=0; i<handlers.size();i++) {
                String key = (String) handlers.get(i);
                if (!key.endsWith(PropertiesConfAccess.DOT 
                		+ PropertiesConfAccess.ACTIVATION)) {
                	w.write(key);
                	w.write(PROP_VAL_SEPS[0]);
                	w.write(props.getProperty(key));
                	w.newLine();
                }
            }
    	}
    	if (loggers.size() > 0) {
        	w.newLine();
        	w.newLine();
        	w.write("#Logger configuration");
        	w.newLine();
            for(int i=0; i<loggers.size();i++) {
                String key = (String) loggers.get(i);
                String value = props.getProperty(key);
                if (key.endsWith(PropertiesConfAccess.DOT 
                		+ PropertiesConfAccess.ADDITIVITY_FIELD) 
                		&& Boolean.valueOf(value).booleanValue()) {
                    continue;
                } else if (key.endsWith(PropertiesConfAccess.DOT 
                		+ PropertiesConfAccess.LEVEL_FIELD) 
                		&& "INHERIT".equalsIgnoreCase(value)) {
                    key = '#' + key;
                }
            	w.write(key);
            	w.write(PROP_VAL_SEPS[0]);
            	w.write(value);
            	w.newLine();
            }
       	}
        w.flush();
    }

    public static Properties readProperties(InputStream is) throws IOException {
        Properties props = new Properties();
        BufferedReader br = new BufferedReader(new InputStreamReader(is));
        while (br.ready()) {
            String p = br.readLine();
            if (p == null || p.length() == 0) {
                continue;
            }
            p = p.replace((char) Character.LINE_SEPARATOR, ' ').trim();
            boolean isComment = false;
            int nc;
            for(nc=0; nc<p.length() && p.charAt(nc) == '#'; nc++) {
                isComment = true;
            }
            if (nc == p.length()) {
                continue;
            }
            if (isComment) {
                p = p.substring(nc).trim();
            }
            int keywordIdx;
            for(keywordIdx=0; keywordIdx<keywords.length; keywordIdx++) {
                if(p.startsWith(keywords[keywordIdx])) {
                    break;
                }
            }
            int sepIdx = -1; 
            if (keywordIdx < keywords.length) {
                for(int i = 0; sepIdx == -1 && i<PROP_VAL_SEPS.length; i++) {
                    sepIdx = p.indexOf(PROP_VAL_SEPS[i]);
                }
            }
            if (sepIdx != -1) {
                String key = p.substring(0, sepIdx);
                String value = p.substring(sepIdx+1).trim();
                if (isComment) {
                    if (keywordIdx == 0 
                    		&& key.endsWith(PropertiesConfAccess.DOT 
                    				+ PropertiesConfAccess.LEVEL_FIELD)) { // logger
                    	props.put(key, "INHERIT");
                    }
                } else {
                	props.put(key, value);
                }
            }
        }
        return props;
    }
}
