/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.objectweb.util.monolog.api.HandlerFactory;
import org.objectweb.util.monolog.api.LevelFactory;
import org.objectweb.util.monolog.api.LoggerFactory;
import org.objectweb.util.monolog.api.MonologFactory;
import org.objectweb.util.monolog.file.monolog.PropertiesConfAccess;
import org.objectweb.util.monolog.wrapper.printwriter.LoggerImpl;

/**
 * This class is a helper to instanciate a logger factory. It provides a set of
 * init(...) method with various parameter:
 *  - no parameter => automatic configuration with system property and default
 *  values
 *  - Properties => contains monolog configuration (loggers, handlers, levels
 *  and optionaly the wrapper class name (automatic choice if not specified)).
 *  - String => the class name of the wrapper class to instanciate
 *
 * @author S.Chassande-Barrioz
 */
public class Monolog {
    /**
     * Inidicates if the monolog wrapper must be logged itself.
     */
    public static boolean debug = new Boolean(
			System.getProperty("monolog.debug")).booleanValue();

    /**
     * is the basic loggerFactory implementation.
     */
    private static MonologFactory basicMonologFactory = new LoggerImpl();

    /**
     * is the last required logger factory. By default it is initialized to the
     * a basic logger factory implementation (LoggerImpl).
     */
    public static MonologFactory monologFactory = basicMonologFactory;

	/**
	 * @deprecated use monologFactory
	 */
	public static LoggerFactory loggerFactory = monologFactory;

    /**
     * is the default monolog configuration file lookup from the file system or
     * from the classpath.
     */
    public static final String DEFAULT_MONOLOG_FILE = "monolog.properties";

    /**
     * is the property name of the monolog configuration file. This property
     * is searched from the specified properties or from the system properties.
     */
    public static final String MONOLOG_FILE_NAME = "monolog.filename";

    /**
     * is the property name of the wrapper class name. This property is searched
     * from the specified properties or from the system properties.
     */
    public static final String MONOLOG_CLASS_NAME = "monolog.classname";

    /**
     * is the class name of logger factory for the log4j logging system.
     */
    public static final String LOG4J_WRAPPER_CLASS_NAME
            = "org.objectweb.util.monolog.wrapper.log4j.MonologLoggerFactory";

    /**
     * is the class name of logger factory for the log4jME logging system.
     */
    public static final String LOG4JMini_WRAPPER_CLASS_NAME
            = "org.objectweb.util.monolog.wrapper.log4jMini.MonologLoggerFactory";

    /**
     * is the class name of logger factory for the java.util.logging logging
     * system.
     */
    public static final String JDK_WRAPPER_CLASS_NAME
            = "org.objectweb.util.monolog.wrapper.javaLog.LoggerFactory";
    /**
     * is the class name used to known if log4j is availlable in the classpath
     */
    private static final String LOG4J_CLASS_NAME = "org.apache.log4j.Logger";

    /**
     * is the class name used to known if log4j is availlable in the classpath
     */
    private static final String LOG4JMini_CLASS_NAME = "org.apache.log4j.Category";

    /**
     * is the class name used to known if the JVM version is greater or equal
     * to 1.4
     */
    private static final String JDK_CLASS_NAME = "java.util.logging.Logger";


	public static MonologFactory getMonologFactory() {
		return monologFactory;
	}

	public static MonologFactory getDefaultMonologFactory() {
		return basicMonologFactory;
	}

    /**
     * Initializes Monolog.
     * It searches in system properties a monolog configuration file
     * (MONOLOG_FILE_NAME system property) or use the default file name
     * (DEFAULT_MONOLOG_FILE system property).
     *
     * @return the loggerFactory (never null)
     */
	public static MonologFactory initialize() {
		if (monologFactory == basicMonologFactory) {
        	getMonologFactory(System.getProperty(MONOLOG_FILE_NAME, DEFAULT_MONOLOG_FILE));
		}
        return monologFactory;
    }

	/**
	 * @deprecated use initialize()
	 */
	public static LoggerFactory init() {
		return initialize();
    }

	/**
	 * @deprecated use getMonologFactory(String)
	 */
	public static LoggerFactory init(String fileName) {
		return getMonologFactory(fileName);
	}

    /**
     * Initializes Monolog with a given monolog configuration file name
     * if the file is found, it delegates the intialization to init(Properties)
     * method. if the file is not found it delegates the logger factory
     * instanciation to the getLoggerFactory(String) method.
     *
     * @param fileName is the file name of a properties reachable from the
     * file system or the classpath.
     * @return the loggerFactory (never null)
     */
    public static MonologFactory getMonologFactory(String fileName) {
        Monolog.debug("DEBUG: debug is activated!");
        File f = new File(fileName);
        InputStream is = null;
        if (f.exists()) {
            Monolog.debug("DEBUG: The file " + fileName
                    + " was found from the file system.");
            try {
                is = new FileInputStream(f);
            } catch (FileNotFoundException e) {
                Monolog.debug("ERROR: " + e.getMessage());
                is = null;
            }
        } else {
            is = Monolog.class.getClassLoader().getResourceAsStream(fileName);
            if (is != null)
                Monolog.debug("DEBUG: The file " + fileName
                        + " was found from the classpath.");
        }
        if (is != null) {
            Properties p = new Properties();
            try {
                p.load(is);
            } catch (IOException e) {
                Monolog.debug("ERROR: problem to load properties from " +
                        "the input stream " + is);
                if (debug) {
                    e.printStackTrace(System.out);
                }
            }
            getMonologFactory(p);
        } else {
            Monolog.debug("DEBUG: The file " + fileName
                    + " was not found.");
            monologFactory = instanciateMonologFactory(null);
        }
        return monologFactory;
    }

    /**
     * Initializes monolog in with a Properties instance.
     * It searches from the properties the logger factory class name (
     * MONOLOG_CLASS_NAME property) and it delegates its configuration to
     * the loadMonologConfiguration(Properties, LoggerFactory) method.
     *
     * @param properties can contains the MONOLOG_CLASS_NAME property and the
     * rest of the monolog configuration (loggers, handlers, levels).
     * @return the loggerFactory (never null)
     */
    public static MonologFactory getMonologFactory(Properties properties) {
        monologFactory = instanciateMonologFactory(properties.getProperty(
                MONOLOG_CLASS_NAME, System.getProperty(MONOLOG_CLASS_NAME)));
        if (monologFactory != basicMonologFactory) {
            loadMonologConfiguration(properties, monologFactory);
        }
        return monologFactory;
    }

	/**
	 * @deprecated use  getMonologFactory(Properties)
	 */
	public static LoggerFactory init(Properties properties) {
		return getMonologFactory(properties);
	}

    /**
     * Loads a monolog configuration into the loggerFactory.
     * @param properties contains the properties to load into the given logger
     * factory.
     */
    public static void loadMonologConfiguration(Properties properties) {
        loadMonologConfiguration(properties, monologFactory);
    }

    /**
     * Loads a monolog configuration into an existing MonologFactory.
     * @param properties contains the properties to load into the given logger
     * factory.
     * @param mf is the MonologFactory to configure.
     */
    public static void loadMonologConfiguration(Properties properties, MonologFactory mf) {
        try {
            PropertiesConfAccess.load(properties, mf, mf, mf);
        } catch (Exception e) {
            Monolog.error("WARN: problem to configure a loggerFactory:" +e.getMessage(), e);
        }
    }
	/**
	 * @deprecated use loadMonologConfiguration(Properties, MonologFactory)
	 */ 
	public static void loadMonologConfiguration(Properties properties, LoggerFactory lf) {
		loadMonologConfiguration(properties, (MonologFactory) lf);
	}

	public static LoggerFactory getLoggerFactory(String className) {
		return getMonologFactory(className);
	}

	public static LevelFactory getLevelFactory(String className) {
		return getMonologFactory(className);
	}

	public static HandlerFactory getHandlerFactory(String className) {
		return getMonologFactory(className);
	}

    /**
     * Retrieves a MonologFactory instance.
     * First it tries to instanciate the given logger factory class name.
     * If it is not possible it tries to instanciate the logger factory of the
     * log4j system (if the classes are availlable).
     * If log4j or its wrapper are not reachable then it tries to instanciate
     * the wrapper of the log4jME logging system.
     * If log4jME or its wrapper are not reachable then it tries to instanciate
     * the wrapper of the java.util.logging logging system.
     * Finally if any of these wrappers are reachable the basic logger factory
     * is used.
     *
     * @param className is the class name of a LoggerFactory implementation.
     * It can be a null value.
     * @return a LoggerFactory instance (never null).
     */
    public static MonologFactory instanciateMonologFactory(String className) {
        //Try with the given logger factory class name if it is not null
        if (className != null) {
            try {
                Monolog.debug("DEBUG: try to use the logger factory: " + className);
                return (MonologFactory) Class.forName(className).newInstance();
            } catch (ClassNotFoundException e) {
                Monolog.debug("WARN: The class " + className
                        + " was not found.");
            } catch (Exception e) {
                Monolog.debug("WARN: The class " + className
                        + " has no public empty constructor.");
            }
        }
        //Try to find the log4j classes and its wrapper
        try {
            Monolog.debug("DEBUG: try to use log4j");
            Class.forName(LOG4J_CLASS_NAME);
            return (MonologFactory)
                    Class.forName(LOG4J_WRAPPER_CLASS_NAME).newInstance();
        } catch (Exception e) {
            Monolog.debug("DEBUG: log4j and its wrapper are not " +
                    "availlable:" + e.getMessage());
            if (debug) {
                e.printStackTrace(System.out);
            }
        }
        //Try to find the jdk classes and its wrapper
        try {
            Monolog.debug("DEBUG: try to use the jdk");
            Class.forName(JDK_CLASS_NAME);
            return (MonologFactory)
                    Class.forName(JDK_WRAPPER_CLASS_NAME).newInstance();
        } catch (Exception e) {
            Monolog.debug("DEBUG: java.util.logging and its wrapper " +
                    "are not availlable:" + e.getMessage());
            if (debug) {
                e.printStackTrace(System.out);
            }
        }

        //Try to find the log4jMini classes and its wrapper
        try {
            Monolog.debug("DEBUG: try to use the log4j mini");
            Class.forName(LOG4JMini_CLASS_NAME);
            return (MonologFactory)
                    Class.forName(LOG4JMini_WRAPPER_CLASS_NAME).newInstance();
        } catch (Exception e) {
            Monolog.debug("DEBUG: log4jMini and its wrapper are not " +
                    "availlable:" + e.getMessage());
            if (debug) {
                e.printStackTrace(System.out);
            }
        }
        Monolog.debug("DEBUG: return the basic logger factory"); 
        //return the basic logger factory
        return basicMonologFactory;
    }
    /**
     * This method must be only used to debug the Monolog wrappers.
     * To active the log of monolog assign the "true" value to the system
     * property "monolog.debug".
     *
     * @param m the message to log.
     */
    static public void debug(String m) {
        if (debug) {
              System.out.println(m);
        }
      }
    static public void error(String m, Exception e) {
        System.err.println(m);
        if (debug) {
            e.printStackTrace(System.err);
        }
      }
}
