/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.api;

/**
 * A Logger implementation receives event messages from an object and exports 
 * them. Each Logger is associated with a log level and discards log requests 
 * that are below this level. Furthermore the Logger interface extends the Handler
 * interface and represents therefore a type of output.
 *
 * @author S.Chassande-Barrioz
 */
public interface Logger extends Handler {

	/**
	 * Permits to set the level with an integer value
	 */
	void setIntLevel(int level);

	/**
	 * Permits to set the level with a Level instance.
	 */
	void setLevel(Level l);

	/**
	 * Returns the current level value under the integer format
	 */
	int getCurrentIntLevel();

	/**
	 * Returns the current level value under the Level format
	 */
	Level getCurrentLevel();
	
	/**
	 * Check if a message of the given level would actually be logged by this logger. 
	 */
	boolean isLoggable(int level);

	/**
	 * Check if a message of the given level would actually be logged by this logger. 
	 */
	boolean isLoggable(Level l);

	/**
	 * Check if this logger is enabled.
	 */
	boolean isOn();
	
	/**
	 * Log a message, with no arguments.
	 * If the logger is currently enabled for the given message level then the
	 * given message is treated
	 */
	void log(int level, Object message);
	/**
	 * Log a message, with no arguments.
	 * If the logger is currently enabled for the given message level then the
	 * given message is treated
	 */
	void log(Level level, Object message);

	/**
	 * Log a message, with a throwable arguments which can represent an 
	 * error or a context..
	 */
	void log(int level, Object message, Throwable throwable);
	/**
	 * Log a message, with a throwable arguments which can represent an 
	 * error or a context..
	 */
	void log(Level level, Object message, Throwable throwable);


	/**
	 * Log a message, with a location and method arguments. The location
	 * parameter can be the object instance which logs the event, or a string
	 * representation of the object.
	 * The method argument can be a java.lang.reflect.Method or a string which
	 * represents the method name.
	 */
	void log(int level, Object message, Object location, Object method);
	/**
	 * Log a message, with a location and method arguments. The location
	 * parameter can be the object instance which logs the event, or a string
	 * representation of the object. 
	 * The method argument can be a java.lang.reflect.Method or a string which
	 * represents the method name.
	 */
	void log(Level l, Object message, Object location, Object method);

	/**
	 * Log a message, with a location, method and throwable arguments. 
	 * The location parameter can be the object instance which logs the 
	 * event, or a string representation of the object.. 
	 * The method argument can be a java.lang.reflect.Method or a string which
	 * represents the method name.
	 * The throwable parameter permits to log an Exception. 
	 */
	void log(int level, Object message, Throwable throwable, 
		Object location, Object method);
	/**
	 * Log a message, with a location, method and throwable arguments. 
	 * The location parameter can be the object instance which logs the 
	 * event, or a string representation of the object.. 
	 * The method argument can be a java.lang.reflect.Method or a string which
	 * represents the method name.
	 * The throwable parameter permits to log an Exception. 
	 */
	void log(Level level, Object message, Throwable throwable, 
		Object location, Object method);

	/**
	 * Enables this logger
	 */
	void turnOn();

	/**
	 * Disables this logger
	 */
	void turnOff();

}
