/* 
 * Licensed to Aduna under one or more contributor license agreements.  
 * See the NOTICE.txt file distributed with this work for additional 
 * information regarding copyright ownership. 
 *
 * Aduna licenses this file to you under the terms of the Aduna BSD 
 * License (the "License"); you may not use this file except in compliance 
 * with the License. See the LICENSE.txt file distributed with this work 
 * for the full License.
 *
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
 * implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package org.openrdf.sail.rdbms.schema;

import java.sql.PreparedStatement;
import java.sql.SQLException;

/**
 * 
 * @author James Leigh
 */
public class Batch {

	public static Batch CLOSED_SIGNAL = new Batch();

	private RdbmsTable temporary;

	private PreparedStatement insertBatch;

	private PreparedStatement insertSelect;

	private int maxBatchSize;

	private int batchCount;

	public int getMaxBatchSize() {
		return maxBatchSize;
	}

	public void setMaxBatchSize(int batchSize) {
		this.maxBatchSize = batchSize;
	}

	public void setTemporary(RdbmsTable temporary) {
		assert temporary != null;
		this.temporary = temporary;
	}

	public void setBatchStatement(PreparedStatement insert) {
		assert insert != null;
		this.insertBatch = insert;
	}

	public void setInsertStatement(PreparedStatement insert) {
		assert insert != null;
		this.insertSelect = insert;
	}

	public int size() {
		return batchCount;
	}

	public boolean isFull() {
		return batchCount >= getMaxBatchSize();
	}

	public boolean isReady() {
		return true;
	}

	public void setObject(int parameterIndex, Object x)
		throws SQLException
	{
		insertBatch.setObject(parameterIndex, x);
	}

	public void setString(int parameterIndex, String x)
		throws SQLException
	{
		insertBatch.setString(parameterIndex, x);
	}

	public void addBatch()
		throws SQLException
	{
		insertBatch.addBatch();
		batchCount++;
	}

	/**
	 * 
	 * @return <code>-1</code> if already flushed
	 * @throws SQLException
	 */
	public int flush()
		throws SQLException
	{
		if (insertBatch == null)
			return -1;
		try {
			int count;
			if (temporary == null) {
				int[] results = insertBatch.executeBatch();
				count = results.length;
			}
			else {
				synchronized (temporary) {
					insertBatch.executeBatch();
					count = insertSelect.executeUpdate();
					temporary.clear();
				}
			}
			return count;
		}
		finally {
			insertBatch.close();
			insertBatch = null;
		}
	}

}
