/* 
 * Licensed to Aduna under one or more contributor license agreements.  
 * See the NOTICE.txt file distributed with this work for additional 
 * information regarding copyright ownership. 
 *
 * Aduna licenses this file to you under the terms of the Aduna BSD 
 * License (the "License"); you may not use this file except in compliance 
 * with the License. See the LICENSE.txt file distributed with this work 
 * for the full License.
 *
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
 * implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package org.openrdf.sail.memory.model;

import org.openrdf.model.Value;

/**
 * A MemoryStore-specific extension of the Value interface, giving it node
 * properties.
 */
public interface MemValue extends Value {

	/*-----------*
	 * Constants *
	 *-----------*/

	/**
	 * A shared empty MemStatementList that is returned by MemURI and MemBNode to
	 * represent an empty list. The use of a shared list reduces memory usage.
	 */
	static final MemStatementList EMPTY_LIST = new MemStatementList(0);

	/*---------*
	 * Methods *
	 *---------*/

	/**
	 * Returns the object that created this MemValue. MemValues are only unique
	 * within a single repository, but an application could use several
	 * repositories at the same time, passing MemValues generated by one Sail to
	 * another Sail. In such situations, the MemValue of the first Sail cannot be
	 * used by the second Sail.
	 */
	public Object getCreator();

	/**
	 * Checks whether this MemValue has any statements. A MemValue object has
	 * statements if there is at least one statement where it is used as the
	 * subject, predicate, object or context value.
	 * 
	 * @return <tt>true</tt> if the MemValue has statements, <tt>false</tt>
	 *         otherwise.
	 */
	public boolean hasStatements();

	/**
	 * Gets the list of statements for which this MemValue is the object.
	 * 
	 * @return A MemStatementList containing the statements.
	 */
	public MemStatementList getObjectStatementList();

	/**
	 * Gets the number of statements for which this MemValue is the object.
	 * 
	 * @return An integer larger than or equal to 0.
	 */
	public int getObjectStatementCount();

	/**
	 * Adds a statement to this MemValue's list of statements for which it is the
	 * object.
	 */
	public void addObjectStatement(MemStatement st);

	/**
	 * Removes a statement from this MemValue's list of statements for which it
	 * is the object.
	 */
	public void removeObjectStatement(MemStatement st);

	/**
	 * Removes statements from old snapshots (those that have expired at or
	 * before the specified snapshot version) from this MemValue's list of
	 * statements for which it is the object.
	 * 
	 * @param currentSnapshot
	 *        The current snapshot version.
	 */
	public void cleanSnapshotsFromObjectStatements(int currentSnapshot);
}
