/*	OMM - Ontology Middleware Module
 *	Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 *	Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *	1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation; either
 *	version 2.1 of the License, or (at your option) any later version.
 *
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	Lesser General Public License for more details.
 *
 *	You should have received a copy of the GNU Lesser General Public
 *	License along with this library; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 */
package org.openrdf.sesame.server.soap;

/**
 * SOAP Services <br>
 * This class provides SOAP Access to OMM/Sesame.
 * <br> Provides access to the RdfSchemaSource, RequestRouter
 * and VersionManagement interfaces.
 * @author damyan ognyanov
 * @author borislav popov
 */

import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.BNodeImpl;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.rio.rdfxml.RdfXmlWriter;

import org.openrdf.sesame.admin.AdminListener;
import org.openrdf.sesame.admin.XmlAdminMsgWriter;
import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.ConfigurationException;
import org.openrdf.sesame.config.RepositoryConfig;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.config.UserInfo;
import org.openrdf.sesame.constants.QueryLanguage;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.omm.SessionContext;
import org.openrdf.sesame.omm.SessionKilled;
import org.openrdf.sesame.omm.VersionManagement;
import org.openrdf.sesame.query.QueryErrorType;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.repository.local.LocalRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.Sail;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.server.SesameServer;

public class SoapServices {

	private final static String VERSION_SAIL = "org.openrdf.sesame.omm.VersionManagement";
	static int lastSessionID = 0;
	private VersionManagement verSail = null;

	private LocalRepository _repository;

	static int createSession()
	{
		return ++lastSessionID;
	}

	int theID;
	SessionKilled sk = null;

	public SoapServices()
	{
		theID = createSession();
	}

	private SessionContext getContext()
	{
		String thisSessionID = "SOAP["+theID+"]";
		SessionContext sc = SessionContext.get(thisSessionID);
		if (sc == null)
		{
			sc = new SessionContext();
			SessionContext.put(thisSessionID, sc);
			System.out.println("new context created ("+thisSessionID+")");
			UserInfo ui = SesameServer.getSystemConfig().getUserInfo(2); // anonymous
			if (ui != null) {
				sc.userID = ui.getID();
				sc.user = ui.getLogin();
				sc.pass = ui.getPassword();
			}
			sk = new SessionKilled(thisSessionID);
		}
		SessionContext.setContext(sc);
		return sc;
	}

	private VersionManagement getVersionManagement() {
		VersionManagement verSail = null;
		// get the versioning sail if such
		try {
			SessionContext sc = getContext();
			
			LocalService service = SesameServer.getLocalService();

			service.login(sc.user, sc.pass);

			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);

			Sail sail = rep.getSail();
			verSail = (VersionManagement)org.openrdf.sesame.sail.util.QuerySailStack.queryInterface( sail, VERSION_SAIL);
		} catch (Exception xx) {
			throw new RuntimeException(xx.getMessage());
		}
		return verSail;
	}


	public int uploadData(String data, String baseURL) {
		SessionContext sc = getContext();
		java.io.OutputStream os = new java.io.ByteArrayOutputStream();
		try {
			AdminListener report = new XmlAdminMsgWriter(os);
			// FIXME the assumption is made that the format is XML
			_repository.addData(data, baseURL, RDFFormat.RDFXML, true, report);
		} catch (AccessDeniedException e) {
			throw new RuntimeException(e.getMessage());
		} catch (java.io.IOException ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return 0;
	}

	public int addDataFromUrl( String dataUrl, String baseURL)
	{
		java.net.URL url = null;
		try {
			// Check existence of URL
			url = new java.net.URL(dataUrl);
		}
		catch (java.net.MalformedURLException e) {
			throw new RuntimeException(e.getMessage());
		}
		
		SessionContext sc = getContext();
		java.io.OutputStream os = new java.io.ByteArrayOutputStream();
		try {
			AdminListener report = new XmlAdminMsgWriter(os);
			// FIXME the assumption is made that the format is XML
			_repository.addData(url, baseURL, RDFFormat.RDFXML, true, report);
		} catch (java.io.IOException e) {
			throw new RuntimeException(e.getMessage());
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage());
		}
		return 0;
	}

	public boolean clearRepository()
	{
		SessionContext sc = getContext();
		java.io.OutputStream os = new java.io.ByteArrayOutputStream();
		try {
			org.openrdf.sesame.admin.AdminListener report = new org.openrdf.sesame.admin.XmlAdminMsgWriter(os);
			_repository.clear(report);
		} catch (java.io.IOException e) {
			throw new RuntimeException(e.getMessage());
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage());
		}
		return true;
	}

	public int removeStatements(String subjURI, String predURI, String objURI, boolean bObjectIsLiteral)
	{
		SessionContext sc = getContext();
		// check input parameters.
		if (subjURI != null && subjURI.trim().equals("")) {
			subjURI = null;
		}
		if (predURI != null && predURI.trim().equals("")) {
			predURI = null;
		}
		if (objURI != null && objURI.trim().equals("")) {
			objURI = null;
		}

		java.io.OutputStream os = new java.io.ByteArrayOutputStream();
		try {
			org.openrdf.sesame.admin.AdminListener report = new org.openrdf.sesame.admin.XmlAdminMsgWriter(os);
			Resource subj = null;

			try {
				subj = new URIImpl(subjURI);
			}
			catch (IllegalArgumentException e) {
				subj = new BNodeImpl(subjURI);
			}

			URI pred = new URIImpl(predURI);

			Value obj = null;

			if (bObjectIsLiteral) {
				obj = new LiteralImpl(objURI);
			}
			else {
				try {
					obj = new URIImpl(objURI);
				}
				catch (IllegalArgumentException e) {
					obj = new BNodeImpl(objURI);
				}
			}
			_repository.removeStatements(subj, pred, obj, report);
		} catch (java.io.IOException e) {
			throw new RuntimeException(e.getMessage());
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage());
		}
		return 0;
	}

	public Vector evalRqlQuery( String query)
		throws Exception
	{
		SessionContext sc = getContext();
		SoapQueryResultListener qrl = new SoapQueryResultListener();
		_repository.performTableQuery(QueryLanguage.RQL, query, qrl);

		return qrl.result();
	}

	public java.util.Vector evalRdqlQuery(String query)
		throws Exception
	{
		SessionContext sc = getContext();
		SoapQueryResultListener qrl = new SoapQueryResultListener();
		_repository.performTableQuery(QueryLanguage.RDQL, query, qrl);

		return qrl.result();
	}

	public String extractRDF(boolean ontology, boolean instances, boolean explicit)
	{
		SessionContext sc = getContext();
		java.io.StringWriter out = new java.io.StringWriter();
		RdfXmlWriter documentWriter = new RdfXmlWriter(out);
		try {
			_repository.extractRDF(documentWriter, ontology, instances, explicit, true);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return out.toString();
	}


	public String[] listRepositories()
	{
		SessionContext sc = getContext();
		/* FIXME request router no longer exists. use a LocalService.
		try {
			return RequestRouter.getRepositoryIDs();
		} catch (AccessDeniedException ex) {
			throw new RuntimeException("AccessDeniedException:"+ex.getMessage());
		} catch (UnknownRepositoryException ex) {
			throw new RuntimeException("UnknownRepositoryException"+ex.getMessage());
		}
		*/
		return new String[0];
	}

	public boolean selectRepository(String repos)
	{
		SessionContext sc = getContext();
		try {
			_repository = (LocalRepository)SesameServer.getLocalService().getRepository(repos);
		} catch (UnknownRepositoryException ex) {
			throw new RuntimeException("UnknownRepositoryException"+ex.getMessage());
		} catch (ConfigurationException ex) {
			throw new RuntimeException("ConfigurationException",ex);
		}
		return true;
	}

	public String getRepositoryTitle(String repositoryId) {
		RepositoryConfig config = SesameServer.getSystemConfig().getRepositoryConfig(repositoryId);
		if ( config == null ) {
			return "";
		}
		return config.getTitle();
	}

	public boolean login(String userID, String pass)
	{
		SessionContext sc = getContext();
		try {
			SesameServer.getLocalService().login(userID, pass);
			return true;
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
	} // login

/**
 *	RdfSchemaSource dellegation
 */
	public Vector getStatements(String subj, String pred, String obj, boolean objIsLiteral)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				Resource subjR = null;
				if (subj != null) {
					subjR = new URIImpl(subj);
				}

				URI predR = null;
				if (pred != null) {
					predR = new URIImpl(pred);
				}

				Value objV = null;
				if (obj != null) {
					if (objIsLiteral) {
						objV = new LiteralImpl(obj);
					}
					else {
						objV = new URIImpl(obj);
					}
				}

				StatementIterator iter = rdfSchema.getStatements(subjR, predR, objV);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} //getStatements()

	public Vector getExplicitStatements(String subj, String pred, String obj, boolean objIsLiteral)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				Resource subjR = null;
				if (subj != null) {
					subjR = new URIImpl(subj);
				}

				URI predR = null;
				if (pred != null) {
					predR = new URIImpl(pred);
				}

				Value objV = null;
				if (obj != null) {
					if (objIsLiteral) {
						objV = new LiteralImpl(obj);
					}
					else {
						objV = new URIImpl(obj);
					}
				}

				StatementIterator iter = rdfSchema.getExplicitStatements(subjR, predR, objV);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} //getExplicitStatements()


	protected Vector _statementIteratorToVector(StatementIterator iter) {
		Vector result = new Vector();

		while (iter.hasNext()) {
			Statement st = iter.next();

			String[] theStatementAsStringArray = new String[3];
			theStatementAsStringArray[0] = _valueToString(st.getSubject());
			theStatementAsStringArray[1] = _valueToString(st.getPredicate());
			theStatementAsStringArray[2] = _valueToString(st.getObject());

			result.add(theStatementAsStringArray);
		}

		return result;
	}

	protected String _valueToString(Value value) {
		if (value instanceof Literal) {
			return ((Literal)value).toString();
		}
		else if (value instanceof URI) {
			return ((URI)value).getURI();
		}
		else if (value instanceof BNode) {
			return ((BNode)value).getID();
		}
		else {
			throw new IllegalArgumentException("value should be a Literal, URI or BNode");
		}
	}


	public boolean hasStatement(String subj, String pred, String obj, boolean objIsLiteral)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				Resource subjR = null;
				if (subj != null) {
					subjR = new URIImpl(subj);
				}

				URI predR = null;
				if (pred != null) {
					predR = new URIImpl(pred);
				}

				Value objV = null;
				if (obj != null) {
					if (objIsLiteral) {
						objV = new LiteralImpl(obj);
					}
					else {
						objV = new URIImpl(obj);
					}
				}

				result = rdfSchema.hasStatement(subjR, predR, objV);
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // hasStatement()

	public boolean hasExplicitStatement(String subj, String pred, String obj, boolean objIsLiteral)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {

			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();
			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				Resource subjR = null;
				if (subj != null) {
					subjR = new URIImpl(subj);
				}

				URI predR = null;
				if (pred != null) {
					predR = new URIImpl(pred);
				}

				Value objV = null;
				if (obj != null) {
					if (objIsLiteral) {
						objV = new LiteralImpl(obj);
					}
					else {
						objV = new URIImpl(obj);
					}
				}

				result = rdfSchema.hasExplicitStatement(subjR, predR, objV);
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // hasExplicitStatement()

	public Vector getClasses()
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {

			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);

			Sail sail = rep.getSail();
			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				StatementIterator iter = rdfSchema.getClasses();
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // getClasses()

	public boolean isClass(String resource)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isClass(new URIImpl(resource));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isClass()

	public Vector getProperties()
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				StatementIterator iter = rdfSchema.getProperties();
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // getProperties()

	public boolean isProperty(String resource)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isProperty(new URIImpl(resource));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isProperty()

	public Vector getSubClassOf(String subClass, String superClass)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				URI rSubClass = null, rSuperClass = null;
				if (subClass != null) {
					rSubClass = new URIImpl(subClass);
				}
				if (superClass != null) {
					rSuperClass = new URIImpl(superClass);
				}

				StatementIterator iter = rdfSchema.getSubClassOf(rSubClass, rSuperClass);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // getSubClassesOf()

	public Vector getDirectSubClassOf(String subClass, String superClass)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				URI rSubClass = null, rSuperClass = null;
				if (subClass != null) {
					rSubClass = new URIImpl(subClass);
				}
				if (superClass != null) {
					rSuperClass = new URIImpl(superClass);
				}

				StatementIterator iter = rdfSchema.getDirectSubClassOf(rSubClass, rSuperClass);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // getDirectSubClassesOf()

	public boolean isSubClassOf(String subClass, String superClass)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isSubClassOf(new URIImpl(subClass), new URIImpl(superClass));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isSubClassOf()

	public boolean isDirectSubClassOf(String subClass, String superClass)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isDirectSubClassOf(new URIImpl(subClass), new URIImpl(superClass));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isDirectSubClassOf()

	public Vector getSubPropertyOf(String subProperty, String superProperty)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				URI rSubProperty = null, rSuperProperty = null;
				if (subProperty != null) {
					rSubProperty = new URIImpl(subProperty);
				}
				if (superProperty != null) {
					rSuperProperty = new URIImpl(superProperty);
				}

				StatementIterator iter = rdfSchema.getSubPropertyOf(rSubProperty, rSuperProperty);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // getSubPropertyesOf()

	public Vector getDirectSubPropertyOf(String subProperty, String superProperty)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				URI rSubProperty = null, rSuperProperty = null;
				if (subProperty != null) {
					rSubProperty = new URIImpl(subProperty);
				}
				if (superProperty != null) {
					rSuperProperty = new URIImpl(superProperty);
				}

				StatementIterator iter = rdfSchema.getDirectSubPropertyOf(rSubProperty, rSuperProperty);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // getDirectSubPropertyesOf()

	public boolean isSubPropertyOf(String subProperty, String superProperty)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isSubPropertyOf(new URIImpl(subProperty), new URIImpl(superProperty));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isSubPropertyOf()

	public boolean isDirectSubPropertyOf(String subProperty, String superProperty)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isDirectSubPropertyOf(new URIImpl(subProperty), new URIImpl(superProperty));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isDirectSubPropertyOf()

	public Vector getType(String anInstance, String aClass)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				URI rInst = null, rClass = null;
				if (anInstance != null) {
					rInst = new URIImpl(anInstance);
				}
				if (aClass != null) {
					rClass = new URIImpl(aClass);
				}

				StatementIterator iter = rdfSchema.getType(rInst, rClass);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} //getType

	public Vector getDirectType(String anInstance, String aClass)
	{
		Vector result = null;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;

				URI rInst = null, rClass = null;
				if (anInstance != null) {
					rInst = new URIImpl(anInstance);
				}
				if (aClass != null) {
					rClass = new URIImpl(aClass);
				}

				StatementIterator iter = rdfSchema.getDirectType(rInst, rClass);
				result = _statementIteratorToVector(iter);
				iter.close();
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} //getDirectType

	public boolean isType(String anInstance, String aClass)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isType(new URIImpl(anInstance), new URIImpl(aClass));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isType()

	public boolean isDirectType(String anInstance, String aClass)
	{
		boolean result = false;
		SessionContext sc = getContext();

		try {
			LocalService service = SesameServer.getLocalService();
			service.login(sc.user, sc.pass);
			LocalRepository rep = (LocalRepository)service.getRepository(sc.repository);
			Sail sail = rep.getSail();

			if (sail instanceof RdfSchemaSource) {
				RdfSchemaSource rdfSchema = (RdfSchemaSource)sail;
				result = rdfSchema.isDirectType(new URIImpl(anInstance), new URIImpl(aClass));
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}

		return result;
	} // isDirectType()

	/* ->->-> Versioning And Tracking ->->-> */

	/**
	 * Create a labeled version for a statte of the repository assigning the
	 * necessary mata-information about thet operation.
	 * @param stateUID the update counter of a particular repository state
	 * @param label the label to be set for this state
	 * NOTE: This method can throw a security exception if the request is made on
	 * behalf of the user with insufficent rights to create versions
	 */
	public void labelState(String stateUID, String label) {
		try {
			verSail = getVersionManagement();
			if (verSail != null) {
				verSail.labelState(Long.parseLong(stateUID),label);
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
	}


	/**
	 * Create a labeled version of the curent repository state.
	 * NOTE: This method can throw an exception if the request is made on
	 * behalf of an user with insufficent rights to create versions
	 */
	public void labelCurrentState(String label) {
		try {
			verSail = getVersionManagement();
			if (verSail != null) {
				verSail.labelCurrentState(label);
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
	}

	/**
	 * Restore the repository to previous state removing all statements added
	 * after the value of the update counter and revive all remover ones.
	 * @param stateUID the update counter of a particular repository state
	 * NOTE: This method can throw an exception if the request is made on
	 * behalf of an user with insufficent rights to revert the repository
	 */
	public void revertToState(String stateUID) {
		SessionContext sc = getContext();
		verSail = getVersionManagement();
		try {
			if (verSail != null) {
				verSail.revertToState(Long.parseLong(stateUID));
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
	}

	/**
	 * Sets the repository to given state for further read operations.
	 * @param stateUID the update counter of a particular repository state
	 */
	public void workWithState(String stateUID){
		verSail = getVersionManagement();
		try {
			if (verSail != null) {
				verSail.workWithState(Long.parseLong(stateUID));
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
	}

	/**
	 * branch the repository at given state for further operations.
	 * @param stateUID the update counter of a particular repository state
	 * @return the id of the repository
	 */
	public String branchState(String stateUID){
		String result = "";
		verSail = getVersionManagement();
		try {
			if (verSail != null) {
				result = verSail.branchState(Long.parseLong(stateUID));
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return result;
	}

	/**
	 * Retrive list of all labeled states of the repository.
	 * @return a list of Versin interfaces for each labeled state of the repository
	 */
	public Vector getVersions(){
		Vector result=null;
		verSail = getVersionManagement();
		try {
			if (verSail != null) {
				Iterator iResult = verSail.getVersions();
				result = new Vector();
				while (iResult.hasNext()) {
					result.add(iResult.next().toString());
				}
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return result;
	}

	/**
	 * Retrive list of all Updates of the repository.
	 * @return a list of Updates of the repository
	 */
	public Vector getUpdateIds() {
//		System.out.println("SoapServices.getUpdateIds()");

		verSail = getVersionManagement();
		Vector result = new Vector();
		try {
			if (verSail != null) {
//				System.out.println("sail instanceof VersionManagement ="+verSail);
				Iterator iResult = verSail.getUpdateIds();
//				System.out.println("Iterator iResult = verSail.getUpdateIds();="+iResult);
				while (iResult.hasNext()) {
					Object o = iResult.next();
					result.add(o);
//					System.out.println(o);
				}
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
//		System.out.println("SoapServices.getUpdateIds() end");
		return result;
	}

	/**
	 * Stop the increment of the update counter. Usefull for a kind of batch updates
	 * or adding a distinct daml coinstructs at once.
	 */
	public void pauseCounterIncrement() {
		verSail = getVersionManagement();

		try {
			if (verSail != null) {
				verSail.pauseCounterIncrement();
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
	}

	/**
	 * Coninue with the normal increment of the update counter on each modification
	 * made in the repository.
	 */
	public void continueCounterIncrement() {
		verSail = getVersionManagement();
		try {
			if (verSail != null) {
				verSail.continueCounterIncrement();
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
	}

	/**
	 * Check if the update couter ss paused
	 * @return true if the updateCouter is paused, flase otherwise
	 */
	public boolean isPausedCounterIncrement() {
		verSail = getVersionManagement();
		boolean result = false;
		try {
			if (verSail != null) {
				result = verSail.isPausedCounterIncrement();
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return result;
	}

	public Vector getVersionIds() {
		verSail = getVersionManagement();
		Vector result = new Vector();
		try {
			if (verSail != null) {
				Iterator iResult = verSail.getVersionIds();
				while (iResult.hasNext()) {
					Object o = iResult.next();
					result.add(o);
				}
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return result;
	}


	public Map getMetaInfo(String subj, String pred, String obj){
		verSail = getVersionManagement();
		Map result = null;
		try {
			if (verSail != null) {
				result = verSail.getMetaInfo(subj,pred,obj);
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return result;
	}

	public Map getVersionMetaInfo(String versionId){
		verSail = getVersionManagement();
		Map result = null;
		try {
			if (verSail != null) {
				result = verSail.getVersionMetaInfo(versionId);
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return result;
	}

	public Map getUpdateMetaInfo(String updateId){
		verSail = getVersionManagement();
		Map result = null;
		try {
			if (verSail != null) {
				result = verSail.getUpdateMetaInfo(updateId);
			}
		} catch (Exception ex) {
			throw new RuntimeException(ex.getMessage());
		}
		return result;
	}

	/* -<-<-< Versioning And Tracking -<-<-< */

} // class SoapServices



class SoapQueryResultListener implements TableQueryResultListener
{
	int cols;
	int current;
	Vector rows;
	String[] row;

	public void startTableQueryResult() throws IOException {
		rows = new Vector();
		cols = 0;
	}

	public void startTableQueryResult(String[] columnHeaders) throws IOException {
		rows = new Vector();
		cols = columnHeaders.length;
	}

	public void endTableQueryResult() throws IOException {
	}

	public void startTuple() throws IOException {
		current = 0;
		row = new String[cols];
	}

	public void endTuple() throws IOException {
		rows.add(row);
	}

	public void tupleValue(Value value) throws IOException {
		row[current++] = value.toString();
	}

	public void error(QueryErrorType errType, String msg) throws IOException {
		System.out.println("Error: "+msg);
	}
	Vector result() {
/*		String[][] ret = new String[rows.size()][];
		for (int i=0; i < rows.size(); i++)
		{
			ret[i] = (String[])rows.get(i);
		}
		*/
		return rows;
	}
}

