/*	OMM - Ontology Middleware Module
 *	Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 *	Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *	1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation; either
 *	version 2.1 of the License, or (at your option) any later version.
 *
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	Lesser General Public License for more details.
 *
 *	You should have received a copy of the GNU Lesser General Public
 *	License along with this library; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 */
package org.openrdf.sesame.server.rmi;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.Map;
import java.util.Vector;

/**
 * ServicesInterface - The interface that allows access to Sesame via RMI.
 * @author Damyan Ognyanoff
 *
 * @deprecated This interface was used in the previous version of the Sesame RMI API
 * @see org.openrdf.sesame.server.rmi.FactoryInterfaceImpl

 */
public interface ServicesInterface extends Remote {

	/**
	 * Uploads RDF(s) in Sesame.
	 * @param data the RDF(s) in a XML form
	 * @param baseURL the base url for the imported data
	 * @return the count of successfuly added statements
	 * @throws RemoteException
	 */
	public int uploadData(String data, String baseURL)
		throws RemoteException;

	/**
	 * Adds RDF(s) given URL of the RDF(s) document and a base URL.
	 * @param dataUrl the URL of the RDF(s) to be imported
	 * @param baseURL the base url for the imported data,
	 * if it is not specified then the dataUrl is used as a base.
	 * @return the count of successfuly added statements
	 * @throws RemoteException
	 */
	public int addDataFromUrl(String dataUrl, String baseURL)
		throws RemoteException;

	/**
	 * Clears the repository.
	 * @return true if the repository has been cleared, otherwise - false
	 * @throws RemoteException
	 */
	public boolean clearRepository()
		throws RemoteException;

	/**
	 * Removes statements. If a parameter xxxURI is null it acts as an asterisk
	 * and indicates all possible values at this position in the triple.
	 * @param subjURI the URI of the subject
	 * @param predURI the URI of the predicate
	 * @param objURI the URI of the object
	 * @param bObjectIsLiteral indicates whether the object is a Literal.
	 * @return the count of statements removed
	 * @throws RemoteException
	 */
	public int removeStatements(String subjURI, String predURI, String objURI, boolean bObjectIsLiteral)
		throws RemoteException;

	/**
	 * Evaluates an RQL Query.
	 * @param query the query to be evaluated
	 * @return the result set as an array of distinct results matching the query,
	 * represented as an array of URIs.
	 * @throws RemoteException
	 */
	public String[][] evalRqlQuery(String query)
		throws RemoteException;

	/**
	 * Evaluates an RDQL query
	 * @param query the query to be evaluated
	 * @return the result set as an array of distinct results matching the query,
	 * represented as an array of URIs.
	 * @throws RemoteException
	 */
	public String[][] evalRdqlQuery(String query)
		throws RemoteException;

	/** Extracts the ontology or/and instances from the repository.
	 * @param ontology whether to include the ontology(i.e. schema) in the result
	 * @param instances whether to include the instances in the result.
	 * @return an XML RDF representation of the ontology(schema) and/or
	 * instances in the repository
	 * @throws RemoteException
	 */
	public String extractRDF(boolean ontology, boolean instances, boolean explicit)
		throws RemoteException;

	/**
	 * Lists all the repositories.
	 * @return an array of repository IDs
	 * @throws RemoteException
	 */
	public String[] listRepositories()
		throws RemoteException;

	/**
	 * Selects a repository to work with.
	 * @param repos repository ID
	 * @return true if the repository was successfully selected
	 * @throws RemoteException
	 */
	public boolean selectRepository(String repos)
		throws RemoteException;

	/**
	 * Logs in given a user and a password
	 * @param userID the id of the user
	 * @param pass the password
	 * @return true if the user was successfully logged in.
	 * @throws RemoteException
	 */
	public boolean login(String userID, String pass)
		throws RemoteException;

	/* ->->-> Versioning And Tracking ->->-> */

	/**
	 * Create a labeled version for a statte of the repository assigning the
	 * necessary mata-information about thet operation.
	 * @param stateUID the update counetr of a particular repository state
	 * @param label the label to be set for this state
	 * NOTE: This method can throw a security exception if the request is made on
	 * behalf of the user with insufficent rights to create versions
	 */
	public void labelState(long stateUID, String label)
		throws RemoteException;

	/**
	 * Create a labeled version of the curent repository state.
	 * NOTE: This method can throw an exception if the request is made on
	 * behalf of an user with insufficent rights to create versions
	 */
	public void labelCurrentState(String label)
		throws RemoteException;

	/**
	 * Restore the repository to previous state removing all statements added
	 * after the value of the update counter and revive all remover ones.
	 * @param stateUID the update counter of a particular repository state
	 * NOTE: This method can throw an exception if the request is made on
	 * behalf of an user with insufficent rights to revert the repository
	 */
	public void revertToState(long stateUID)
		throws RemoteException;


	/**
	 * Sets the repository to given statete for further read operations.
	 * @param stateUID the update counter of a particular repository state
	 */
	public void workWithState(long stateUID)
		throws RemoteException;


	/**
	 * branch the repository at given state for further operations.
	 * @param stateUID the update counter of a particular repository state
	 * @return the id of the repository
	 */
	public String branchState(long stateUID)
		throws RemoteException;


	/**
	 * Retrive list of all labeled states of the repository.
	 * @return a list of Versin interfaces for each labeled state of the repository
	 */
	public Vector getVersions()
		throws RemoteException;

	/**
	 * Retrive list of all Updates of the repository.
	 * @return a list of Updates of the repository
	 */
	public Vector getUpdateIds()
		throws RemoteException;


	/**
	 * Stop the increment of the update counter. Usefull for a kind of batch updates
	 * or adding a distinct daml coinstructs at once.
	 */
	public void pauseCounterIncrement()
		throws RemoteException;


	/**
	 * Coninue with the normal increment of the update counter on each modification
	 * made in the repository.
	 */
	public void continueCounterIncrement()
		throws RemoteException;


	/**
	 * Check if the update couter ss paused
	 * @return true if the updateCouter is paused, flase otherwise
	 */
	public boolean isPausedCounterIncrement()
		throws RemoteException;

	public Vector getVersionIds()
		throws RemoteException;

	public Map getMetaInfo(String subj, String pred, String obj)
		throws RemoteException;

	public Map getVersionMetaInfo(String versionId)
		throws RemoteException;

	public Map getUpdateMetaInfo(String updateId)
		throws RemoteException;


	/* -<-<-< Versioning And Tracking -<-<-< */

 /* ->->-> SAIL ->->-> */

 /**
	* Retrieve a list of the defined classes in the repository
	* @return Vector containig the URI's of the classes
	*/
	public Vector getClasses()
		throws RemoteException ;

	/**
	 * Retrieve a list of URIs of an instances about specific class
	 * @param aClass Class to explore
	 * @return Vector containig the URIs of the instances
	 */
	public Vector getType(String anInstance, String aClass)
		throws RemoteException;

	/**
	 * Retrieve a list of URIs of an instances about specific class
	 * @param aClass Class to explore
	 * @return Vector containig the URIs of the instances
	 */
	public Vector getDirectType(String anInstance, String aClass)
		throws RemoteException;

	/**
	 * Retrieve a list of the defined properties in the repository
	 * @return Vector containig the URI's of the properties
	 */
	public Vector getProperties()
		throws RemoteException;

	/**
	 * Retrive a list of the statemnets from the repository. All the
	 * statements that match the provided pattern will be included into teh result
	 * @param subj URI of the subject, The null means that the subject does not mather
	 * and the statement will be included regarding on the match of its other
	 * componets(predicate, object, or is it an explicit one)
	 * @param pred URI of the predicate. Tha same rule for the null value is valid
	 * @param obj URI of the object. Tha same rule for the null value is valid
	 * @param objIsLiteral indicating the the URI of the object is actually an Literal not an
	 * Resource URI.
	 * @return Vector containig the String[] with URI's of the distinct componets
	 * of each statement (subject, predicate and object)
	 */
	public Vector getStatements(String subj, String pred, String obj, boolean objIsLiteral)
	throws RemoteException;

	/**
	 * Retrive a list of the explicit statemnets from the repository. All the
	 * explicit statements that match the provided pattern will be included into teh result
	 * @param subj URI of the subject, The null means that the subject does not mather
	 * and the statement will be included regarding on the match of its other
	 * componets(predicate, object, or is it an explicit one)
	 * @param pred URI of the predicate. Tha same rule for the null value is valid
	 * @param obj URI of the object. Tha same rule for the null value is valid
	 * @param objIsLiteral indicating the the URI of the object is actually an Literal not an
	 * Resource URI.
	 * @return Vector containig the String[] with URI's of the distinct componets
	 * of each statement (subject, predicate and object)
	 */
	public Vector getExplicitStatements(String subj, String pred, String obj, boolean objIsLiteral)
	throws RemoteException;

	/**
	 * Query the repository for particular statement
	 * @param subj URI of Subject
	 * @param pred URI of Predicate
	 * @param obj URI of Object
	 * @param objIsLiteral flag indicationg the object URI is actuallt a Literal
	 */
	public boolean hasStatement(String subj, String pred, String obj, boolean objIsLiteral)
	throws RemoteException ;

	/**
	 * Query the repository for particular explicit statement
	 * @param subj URI of Subject
	 * @param pred URI of Predicate
	 * @param obj URI of Object
	 * @param objIsLiteral flag indicationg the object URI is actuallt a Literal
	 */
	public boolean hasExplicitStatement(String subj, String pred, String obj, boolean objIsLiteral)
	throws RemoteException ;

	/**
	 * Retrieve a list of the subclasses of a class
	 * @param subClass URI of the subclass
	 * @param superClass URI of the superclass
	 * @return Vector containing the URIs of the subclasses
	 */
	public Vector getSubClassOf(String subClass, String superClass)
	throws RemoteException;

	/**
	 * Retrieve a list of the direct subclasses of a class
	 * @param subClass URI of the subclass
	 * @param superClass URI of the superclass
	 * @return Vector containing the URIs of the subclasses
	 */
	public Vector getDirectSubClassOf(String subClass, String superClass)
	throws RemoteException;

	/**
	 * Retrieve a list of the subproperties of a property
	 * @param subProperty URI of the subproperty
	 * @param superProperty URI of the superproperty
	 * @return Vector containing the URIs of the subproperties
	 */
	public Vector getSubPropertyOf(String subProperty, String superProperty)
	throws RemoteException;

	/**
	 * Retrieve a list of the direct subproperties of a property
	 * @param subProperty URI of the subproperty
	 * @param superProperty URI of the superproperty
	 * @return Vector containing the URIs of the subproperties
	 */
	public Vector getDirectSubPropertyOf(String subProperty, String superProperty)
	throws RemoteException;

	/**
	 * Check if an URI is a class
	 * @param resource URI to check
	 * @return true if the resource is class, otherwise - null
	 */
	public boolean isClass(String resource)throws RemoteException;

	/**
	 * Check if an URI is a property
	 * @param resource URI to check
	 * @return true if the resource is a property, otherwise -false.
	 */
	public boolean isProperty(String resource)
		throws RemoteException;

	/**
	 * Check if an URI is an instance of a class
	 * @param anInstance URI of the instance to check
	 * @param aClass URI of the class to check for
	 * @return true if not fail
	 */
	public boolean isType(String anInstance, String aClass)
	throws RemoteException;

	/**
	 * Check if an URI is a direct instance of a class
	 * @param anInstance URI of the instance to check
	 * @param aClass URI of the class to check for
	 * @return true if not fail
	 */
	public boolean isDirectType(String anInstance, String aClass)
	throws RemoteException;

	/**
	 * Query for subsumption of two classes
	 * @param subClass URI of the subClass
	 * @param superClass URI of the superClass
	 * @return true if subsumes, otherwise - false.
	 */
	public boolean isSubClassOf(String subClass, String superClass)
		throws RemoteException;

	/**
	 * Query for direct subsumption of two classes
	 * @param subClass URI of the subClass
	 * @param superClass URI of the superClass
	 * @return true if direct subsumes, otherwise - false.
	 */
	public boolean isDirectSubClassOf(String subClass, String superClass)
		throws RemoteException;

	/**
	 * Query for subsumption of two properties.
	 * @param subProperty URI of the subProperty
	 * @param superProperty URI of the superProperty
	 * @return true - if a subproperty, otherwise - false.
	 */
	public boolean isSubPropertyOf(String subProperty, String superProperty)
		throws RemoteException;

	/**
	 * Query for direct subsumption of two properties.
	 * @param subProperty URI of the subProperty
	 * @param superProperty URI of the superProperty
	 * @return true - if a direct subproperty, otherwise - false.
	 */
	public boolean isDirectSubPropertyOf(String subProperty, String superProperty)
		throws RemoteException;


	/* -<-<-< SAIL <-<-<- */

	/* OTHER NOT DOCUMENTED */

	public String getRepositoryTitle(String repositoryId)
		throws RemoteException;



} // interface ServicesInterface
