/*  OMM - Ontology Middleware Module
 *  Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 *  Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *  1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.server.rmi;


import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;

import org.openrdf.util.rmirouting.ChannelIfaceImpl;
import org.openrdf.util.rmirouting.KeepAliveWhenUnreferenced;
import org.openrdf.util.rmirouting.RoutingAction;

import org.openrdf.sesame.repository.SesameService;
import org.openrdf.sesame.server.SesameServer;

public class FactoryInterfaceImpl implements FactoryInterface, KeepAliveWhenUnreferenced {

  private static String sessionPrefix = String.valueOf(System.currentTimeMillis());
  private static int sessionCounter = 1;
/**
 * bind() - register the first and only FactoryInterface instance to the
 * RMI registry on the localhost. The procedureis the following:
 *  - tries to get a ref to the running registry on the default port
 *  - make attemt to bind or either rebind an instance to it
 *  - if the bind/rebind fails the it tries to create a new registry on the
 *  localhost listening on the default port
 *  - make a new attamt to bind the new FactoryInterface instance to that registry
 */
  public static void bind() {
    bind(new Integer(java.rmi.registry.Registry.REGISTRY_PORT));
  }
  public static void bind(Integer port) {
    boolean bNotError = true;
    try {
// retrieve a ref to the currently running RMI registry
      Registry reg = LocateRegistry.getRegistry(port.intValue());
      if (reg == null) {
        reg = LocateRegistry.createRegistry(port.intValue());
      }
// create an instance fro binding
      FactoryInterface theFactory = new FactoryInterfaceImpl();
      ChannelIfaceImpl.setPrologAction(
          new RoutingAction() {
            public void doRoutingAction(Object cookie) {
              ServicesInterfaceImpl.getContext(cookie);
            }
          }
        );
      Remote fi = ChannelIfaceImpl.createStub(theFactory);
//bind/rebind
      try {
        reg.bind("FactoryInterface", fi);
      } catch (java.rmi.AlreadyBoundException e ) {
          reg.rebind("FactoryInterface", fi);
      } catch (java.rmi.ConnectException e ) {
// if such exception occurs we can try again with a newly created registry(using defaults)
          reg = LocateRegistry.createRegistry(port.intValue());
// bind/rebind
          try {
            reg.bind("FactoryInterface", fi);
          } catch (java.rmi.AlreadyBoundException e1 ) {
            reg.rebind("FactoryInterface", fi);
          }
      }
    } catch (RemoteException re) {
    //failed
      System.out.println("RMI:FactoryInterfaceImpl:RemoteException:"+re.getMessage());
      bNotError = false;
    }
// report success/failure status
    if (bNotError)
      System.out.println("FactoryInteface registered on port "+port);
    else
      System.out.println("FactoryInteface is not registered");
  }

/**
 * unbind() - deregister the only FactoryInterface instance from the running
 * RMI registry
 */
  public static void unbind(){
    unbind(new Integer(java.rmi.registry.Registry.REGISTRY_PORT));
  }
  public static void unbind(Integer port){
//retrieve the ref to the running RMI registry
    try {
      Registry reg = LocateRegistry.getRegistry(port.intValue());
      if (reg != null) {
// unbind it
        reg.unbind("FactoryInterface");
        System.out.println("FactoryInteface unregistered from port "+port);
      }
    } catch (RemoteException re) {
    // something failed
      System.out.println("FactoryInteface failed:"+re.getMessage());
    } catch (java.rmi.NotBoundException re) {
      System.out.println("FactoryInteface failed:"+re.getMessage());
    }
  }

  /**
   * creates a unique ID used to associate and identify the context instance for
   * any of the created services
   * @return the unique String value
   */
  private String nextSessionID() {return sessionPrefix+String.valueOf(sessionCounter++);}

  public FactoryInterfaceImpl() throws RemoteException {
    super();
  }
/**
 * getService() - create an instance implementing the ServicesInterface.
 * @return the new instance
 */
  public SesameService getService() {
    org.openrdf.sesame.omm.SessionContext.setContext(null);
    String cookieTouse = nextSessionID();
    ChannelIfaceImpl.setCurrentCookie(cookieTouse);
    return new ServicesInterfaceImpl(cookieTouse);
  }

  public void stopService() {
    SesameServer.clear();
    new java.util.Timer().schedule(
      // clean up on a sepearate thread
      new java.util.TimerTask() {
              public void run() {
                              System.exit(0);
              }
      } , 3000);
  }
  /**
   * test bind/unbind code
   */
  public static void main(String[] options) {
    FactoryInterfaceImpl.bind();
    FactoryInterfaceImpl.unbind();
  }
}
