/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.QueryLanguage;
import org.openrdf.sesame.constants.QueryResultFormat;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.query.BinaryTableResultWriter;
import org.openrdf.sesame.query.HtmlTableWriter;
import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryErrorType;
import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.query.VarBindingRdfWriter;
import org.openrdf.sesame.query.XmlQueryResultWriter;
import org.openrdf.sesame.repository.SesameRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPErrorType;
import org.openrdf.sesame.server.SesameServer;

public class TableQueryServlet extends SesameServlet {

	protected void _doGet(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		_handleFormURLEncodedRequest(request, response);
	}

	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (x-www-form-urlencoded)
		String repository = HttpServerUtil.getParameter(request, "repository");
		String query = HttpServerUtil.getParameter(request, "query");
		String queryLanguageStr = HttpServerUtil.getParameter(request, "queryLanguage");
		String resultFormatStr = HttpServerUtil.getParameter(request, "resultFormat", QueryResultFormat.XML.toString());
		String serializationStr = HttpServerUtil.getParameter(request, "serialization", RDFFormat.RDFXML.toString());
		String plainXMLLiteralsStr = HttpServerUtil.getParameter(request, "plainXMLLiterals", "off");

		_handleRequest(request, response, repository, query, queryLanguageStr,
						resultFormatStr, serializationStr, plainXMLLiteralsStr);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (multipart/form-data)
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		String repository = HttpServerUtil.getParameter(fileItemMap, "repository");
		String query = HttpServerUtil.getParameter(fileItemMap, "query");
		String queryLanguageStr = HttpServerUtil.getParameter(fileItemMap, "queryLanguage");
		String resultFormatStr = HttpServerUtil.getParameter(fileItemMap, "resultFormat", QueryResultFormat.XML.toString());
		String serializationStr = HttpServerUtil.getParameter(fileItemMap, "serialization", RDFFormat.RDFXML.toString());
		String plainXMLLiteralsStr = HttpServerUtil.getParameter(fileItemMap, "plainXMLLiterals", "off");

		_handleRequest(request, response, repository, query, queryLanguageStr,
						resultFormatStr, serializationStr, plainXMLLiteralsStr);
	}

	private void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String repository, String query, String queryLanguageStr,
		String resultFormatStr, String serializationStr, String plainXMLLiteralsStr)
		throws IOException
	{
		SesameServer.setThreadLogFileForRepository(repository);
		_logIP(request);
		ThreadLog.log(">>> evaluate table query");

		// trace output
		ThreadLog.trace("query = " + query);
		ThreadLog.trace("queryLanguage = " + queryLanguageStr);
		ThreadLog.trace("repository = " + repository);
		ThreadLog.trace("resultFormat = " + resultFormatStr);
		ThreadLog.trace("serialization = " + serializationStr);
		ThreadLog.trace("plainXMLLiterals = " + plainXMLLiteralsStr);

		// Check request parameters
		if (query == null) {
			_sendBadRequest("Query is missing", response);
			return;
		}

		if (repository == null) {
			_sendBadRequest("No repository specified", response);
			return;
		}

		if (queryLanguageStr == null) {
			_sendBadRequest("Query language not specified", response);
			return;
		}

		QueryLanguage queryLanguage = QueryLanguage.forValue(queryLanguageStr);
		if (queryLanguage == null) {
			_sendBadRequest("Unsupported query language: " + queryLanguageStr, response);
			return;
		}

		boolean plainXMLLiterals = plainXMLLiteralsStr.equals("on");

		TableQueryResultListener resListener = null;
		QueryResultFormat resultFormat = QueryResultFormat.forValue(resultFormatStr);

		HTTPOutputStream httpOut = new HTTPOutputStream(response);
		httpOut.setUseGZIPCompression(HttpServerUtil.acceptsGZIPEncoding(request));
		httpOut.setCacheableResult(false);

		if (resultFormat == QueryResultFormat.XML) {
			httpOut.setContentType("text/xml; charset=utf-8");
 		  	resListener = new XmlQueryResultWriter(httpOut, plainXMLLiterals);
		}
		else if (resultFormat == QueryResultFormat.HTML) {
			httpOut.setContentType("text/html");
			String styleSheetURL = request.getContextPath() + "/query.css";
			resListener = new HtmlTableWriter(httpOut, repository, styleSheetURL);
		}
		else if (resultFormat == QueryResultFormat.RDF) {
			// Check serialization format
			RDFFormat serialization = RDFFormat.forValue(serializationStr);

			if (serialization == RDFFormat.RDFXML) {
				httpOut.setContentType("application/rdf+xml; charset=utf-8");
				resListener = new VarBindingRdfWriter(serialization, httpOut);
			}
			else if (serialization == RDFFormat.TURTLE) {
				httpOut.setContentType("application/x-turtle");
				resListener = new VarBindingRdfWriter(serialization, httpOut);
			}
			else if (serialization == RDFFormat.NTRIPLES) {
				httpOut.setContentType("text/plain");
				resListener = new VarBindingRdfWriter(serialization, httpOut);
			}
			else if (serialization == RDFFormat.N3) {
				httpOut.setContentType("text/rdf+n3");
				resListener = new VarBindingRdfWriter(serialization, httpOut);
			}
			else {
				_sendBadRequest("Unknown serialization format: " + serializationStr, response);
				return;
			}
		}
		else if (resultFormat == QueryResultFormat.BINARY) {
			httpOut.setContentType("application/x-binary-rdf-results-table");
			resListener = new BinaryTableResultWriter(httpOut);
		}
		else {
			_sendBadRequest("Unknown result format: " + resultFormatStr, response);
			return;
		}

		try {
			LocalService service = SesameServer.getLocalService();

			_login(service);

			SesameRepository rep = service.getRepository(repository);

			// Evaluate the query
			rep.performTableQuery(queryLanguage, query, resListener);
			httpOut.close();
		}
		catch (UnknownRepositoryException e) {
			_sendBadRequest(HTTPErrorType.UNKNOWN_REPOSITORY, "Unknown repository: " + repository, response);
		}
		catch (AccessDeniedException e) {
			_sendForbidden(e.getMessage(), response);
		}
		catch (MalformedQueryException e) {
			_sendBadRequest(HTTPErrorType.MALFORMED_QUERY, e.getMessage(), response);
		}
		catch (QueryEvaluationException e) {
			ThreadLog.warning("Query evaluation error: " + e.getMessage());
			if (!httpOut.streamOpened()) {
				_sendBadRequest(HTTPErrorType.QUERY_EVALUATION_ERROR, e.getMessage(), response);
			}
			else {
				// Data has already been sent to the client, we can't send an
				// HTTP error code anymore.
				resListener.error(QueryErrorType.QUERY_EVALUATION_ERROR, e.getMessage());
				resListener.endTableQueryResult();
			}
		}
		catch (IOException e) {
			ThreadLog.warning("I/O error, client connection closed?\n" + e.getMessage());
		}
		catch (Exception e) {
			ThreadLog.error("Unknown error on query: " + query, e);

			resListener.error(QueryErrorType.QUERY_EVALUATION_ERROR,
						"The server generated an unknown error. Please inform the server administrator.");
			resListener.endTableQueryResult();
		}
	}
}
