/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.omm.SessionContext;
import org.openrdf.sesame.omm.SessionKilled;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPErrorType;
import org.openrdf.sesame.server.SesameServer;

public abstract class SesameServlet extends HttpServlet {

/*--------------+
| Constants     |
+--------------*/

	protected static final String SESSIONID = "sesame_sid";

/*--------------------------+
| Methods from HttpServlet  |
+--------------------------*/

	/**
	 *
	 **/
	public final void init(ServletConfig config)
		throws ServletException
	{
		super.init(config);
		_init(config);
		SesameServer.unsetThreadLogFile();
	}

	protected void _init(ServletConfig config)
		throws ServletException
	{
	}

	/**
	 * Release resources.
	 **/
	public final void destroy() {
		_destroy();
		SesameServer.unsetThreadLogFile();
		super.destroy();
	}

	protected void _destroy()
	{
	}

	public final void doGet(
		HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		long startTime = System.currentTimeMillis();

		setSessionContext(request);

		_doGet(request, response);

		unsetSessionContext();

		long endTime = System.currentTimeMillis();
		ThreadLog.log(">>> request processed in " + (endTime - startTime) + " ms");

		SesameServer.unsetThreadLogFile();
	}

	protected void _doGet(
		HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		super.doGet(request, response);
	}

	public final void doPost(
		HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		long startTime = System.currentTimeMillis();

		setSessionContext(request);

		_doPost(request, response);

		unsetSessionContext();

		long endTime = System.currentTimeMillis();
		ThreadLog.log(">>> request processed in " + (endTime - startTime) + " ms");

		SesameServer.unsetThreadLogFile();
	}

	protected void _doPost(
		HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		super.doPost(request, response);
	}

/*--------------------------------------------+
|	Methods concerning SessionContext         |
+--------------------------------------------*/

	public static void setSessionContext(HttpServletRequest request)
		throws ServletException, IOException
	{
		// Clear current thread info to prevent anonymous users from gaining
		// access to someone else's repository when threads are re-used:
		unsetSessionContext();

		Cookie[] cookies = request.getCookies();
		Cookie sessionCookie = null;
		if (cookies != null) {
			for (int i = 0; i < cookies.length; i++) {
				if (cookies[i].getName().equals(SESSIONID)) {
					sessionCookie = cookies[i];
					break;
				}
			}
		}

		if (sessionCookie != null) {
			SessionContext sc = SessionContext.get(sessionCookie.getValue());

			if (sc == null) {
				sc = new SessionContext();
				SessionContext.put(sessionCookie.getValue(), sc);

				HttpSession session = request.getSession();
				session.setAttribute("foo", new SessionKilled(sessionCookie.getValue()));
			}
			else {
				ThreadLog.trace("session context: user=" + sc.user + "; repository=" + sc.repository);
			}

			SessionContext.setContext(sc);
		}
	}

	public static void unsetSessionContext()
		throws ServletException, IOException
	{
		SessionContext.setContext(null);
	}

/*--------------------------------------------+
|	Utlity methods for subclasses             |
+--------------------------------------------*/

	/**
	 * Logs the request's IP address to <tt>ThreadLog</tt>.
	 **/
	protected void _logIP(HttpServletRequest request) {
		ThreadLog.log(">>> request from " + request.getRemoteAddr());
	}

	/**
	 * Tries to log in to the supplied <tt>LocalService</tt> with the
	 * credentials that are available through <tt>SessionContext</tt>.
	 *
	 * @param service The LocalService that should be logged in to.
	 * @exception AccessDeniedException If the attempt to log in failed.
	 **/
	protected void _login(LocalService service)
		throws AccessDeniedException
	{
		SessionContext sc = SessionContext.getContext();
		if (sc != null && sc.user.trim().length() > 0) {
			ThreadLog.trace("user = " + sc.user);
			service.login(sc.user, sc.pass);
		}
	}

	/**
	 * Sends a <tt>BAD REQUEST</tt> response with the supplied message.
	 **/
	protected void _sendBadRequest(String msg, HttpServletResponse response)
		throws IOException
	{
		ThreadLog.log("Bad request: " + msg);
		response.sendError(HttpServletResponse.SC_BAD_REQUEST, msg);
	}

	/**
	 * Sends a <tt>BAD REQUEST</tt> response with the supplied error type
	 * and message.
	 **/
	protected void _sendBadRequest(HTTPErrorType errType, String msg, HttpServletResponse response)
		throws IOException
	{
		ThreadLog.log("Bad request: " + msg);
		response.sendError(HttpServletResponse.SC_BAD_REQUEST, errType.toString() + ": " + msg);
	}

	/**
	 * Sends a <tt>FORBIDDEN</tt> response with the supplied message.
	 **/
	protected void _sendForbidden(String msg, HttpServletResponse response)
		throws IOException
	{
		ThreadLog.log("Access denied: " + msg);
		response.sendError(HttpServletResponse.SC_FORBIDDEN, msg);
	}

	/**
	 * Sends an <tt>INTERNAL SERVER ERROR</tt> response with the
	 * supplied message and also logs the error to <tt>ThreadLog</tt>.
	 **/
	protected void _sendInternalError(String msg, Exception e, HttpServletResponse response)
		throws IOException
	{
		ThreadLog.error("Internal error: " + msg, e);
		response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, msg);
	}
}
