/*
 * Sesame - Storage and Querying architecture for RDF and RDF Schema 
 * Copyright (C) 2001-2006 Aduna
 * 
 * Contact: 
 * 	Aduna 
 *  Prinses Julianaplein 14 b 
 *  3817 CS Amersfoort 
 *  The Netherlands
 *  tel. +33 (0)33 465 99 87 
 *  fax. +33 (0)33 465 99 87
 * 
 * http://aduna-software.com/ 
 * http://www.openrdf.org/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.ValueFactoryImpl;

import org.openrdf.rio.Parser;
import org.openrdf.rio.StatementHandler;
import org.openrdf.rio.StatementHandlerException;
import org.openrdf.rio.ntriples.NTriplesParser;
import org.openrdf.rio.ntriples.NTriplesUtil;
import org.openrdf.rio.rdfxml.RdfXmlParser;
import org.openrdf.rio.turtle.TurtleParser;

import org.openrdf.sesame.admin.AdminListener;
import org.openrdf.sesame.admin.HtmlAdminMsgWriter;
import org.openrdf.sesame.admin.XmlAdminMsgWriter;
import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.AdminResultFormat;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.repository.local.LocalRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPErrorType;
import org.openrdf.sesame.sail.RdfRepository;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.server.SesameServer;

public class RemoveStatementsServlet extends SesameServlet {

	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters
		String repository = HttpServerUtil.getParameter(request, "repository");
		String resultFormatStr = HttpServerUtil.getParameter(request, "resultFormat", AdminResultFormat.XML.toString());

		// subject/predicate/object pattern
		String subject = HttpServerUtil.getParameter(request, "subject");
		String predicate = HttpServerUtil.getParameter(request, "predicate");
		String object = HttpServerUtil.getParameter(request, "object");

		// arbitrary graph
		String data = HttpServerUtil.getParameter(request, "data");
		String baseURI = HttpServerUtil.getParameter(request, "baseURI", "foo:bar");
		String dataFormatStr = HttpServerUtil.getParameter(request, "dataFormat");

		_handleRequest(request, response, repository, resultFormatStr, subject, predicate, object,
						data, baseURI, dataFormatStr);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (multipart/form-data)
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		// Get request parameters
		String repository = HttpServerUtil.getParameter(fileItemMap, "repository");
		String resultFormatStr = HttpServerUtil.getParameter(fileItemMap, "resultFormat", AdminResultFormat.XML.toString());

		// subject/predicate/object pattern
		String subject = HttpServerUtil.getParameter(fileItemMap, "subject");
		String predicate = HttpServerUtil.getParameter(fileItemMap, "predicate");
		String object = HttpServerUtil.getParameter(fileItemMap, "object");

		// arbitrary graph
		String data = HttpServerUtil.getParameter(fileItemMap, "data");
		String baseURI = HttpServerUtil.getParameter(fileItemMap, "baseURI", "foo:bar");
		String dataFormatStr = HttpServerUtil.getParameter(fileItemMap, "dataFormat");

		_handleRequest(request, response, repository, resultFormatStr, subject, predicate, object,
						data, baseURI, dataFormatStr);
	}

	private void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String repository, String resultFormatStr, String subject, String predicate, String object,
		String data, String baseURI, String dataFormatStr)
		throws IOException
	{
		SesameServer.setThreadLogFileForRepository(repository);
		_logIP(request);
		ThreadLog.log(">>> remove statements");

		// Log parameters
		ThreadLog.trace("repository = " + repository);
		ThreadLog.trace("resultFormat = " + resultFormatStr);
		ThreadLog.trace("subject = " + subject);
		ThreadLog.trace("predicate = " + predicate);
		ThreadLog.trace("object = " + object);
		ThreadLog.trace("data file sent: " + (data != null));
		ThreadLog.trace("dataFormat = " + dataFormatStr);

		// Check repository parameter
		if (repository == null) {
			_sendBadRequest("No repository specified", response);
			return;
		}

		// Parse the N-Triples values
		ValueFactory valFactory = new ValueFactoryImpl();
		Resource subj = null;
		URI pred = null;
		Value obj = null;
		String errMsg = null;

		try {
			if (subject != null) {
				errMsg = "Illegal value for subject: " + subject;
				subj = NTriplesUtil.parseResource(subject, valFactory);
			}
			if (predicate != null) {
				errMsg = "Illegal value for predicate: " + predicate;
				pred = NTriplesUtil.parseURI(predicate, valFactory);
			}
			if (object != null) {
				errMsg = "Illegal value for object: " + object;
				obj = NTriplesUtil.parseValue(object, valFactory);
			}
		}
		catch (IllegalArgumentException e) {
			_sendBadRequest(errMsg, response);
			return;
		}

		// Check the result format and create an AdminListener
		HTTPOutputStream httpOut = new HTTPOutputStream(response);
		httpOut.setCacheableResult(false);

		AdminResultFormat resultFormat = AdminResultFormat.forValue(resultFormatStr);
		AdminListener adminListener = null;

		if (resultFormat == AdminResultFormat.XML) {
			httpOut.setContentType("text/xml");
			adminListener = new XmlAdminMsgWriter(httpOut);
		}
		else if (resultFormat == AdminResultFormat.HTML) {
			httpOut.setContentType("text/html");
			adminListener = new HtmlAdminMsgWriter(httpOut);
		}
		else {
			_sendBadRequest("Unknown result format: " + resultFormatStr, response);
			return;
		}

		RdfRepository rdfRepository = null;

		try {
			LocalService service = SesameServer.getLocalService();

			_login(service);

			LocalRepository localRepository = (LocalRepository)service.getRepository(repository);
			rdfRepository = (RdfRepository)localRepository.getSail();

			if (data != null) {
				// remove all statements that are in the document

				// check data format
				if (dataFormatStr == null) {
					_sendBadRequest("no data serialization format specified", response);
					return;
				}

				RDFFormat dataFormat = RDFFormat.forValue(dataFormatStr);
				Parser parser = null;

				if (dataFormat.equals(RDFFormat.TURTLE)) {
					parser = new TurtleParser();
				}
				else if (dataFormat.equals(RDFFormat.RDFXML)) {
					parser = new RdfXmlParser();
				}
				else if (dataFormat.equals(RDFFormat.NTRIPLES)) {
					parser = new NTriplesParser();
				}
				else {
					_sendBadRequest("unknown data serialization format: " + dataFormatStr, response);
					return;
				}

				parser.setStatementHandler( new StatementRemover(rdfRepository) );
				parser.setParseErrorListener( new ErrorHandler(adminListener) );
				parser.setPreserveBNodeIds(true);

				Reader reader = new StringReader(data);

				rdfRepository.startTransaction();
				adminListener.transactionStart();
				parser.parse(reader, baseURI);
			}
			else {
				// Remove the statements
				long startTime = System.currentTimeMillis();

				localRepository.removeStatements(subj, pred, obj, adminListener);

				long endTime = System.currentTimeMillis();
				ThreadLog.trace("removed in " + (endTime - startTime) + "ms");
			}
		}
		catch (UnknownRepositoryException e) {
			_sendBadRequest(HTTPErrorType.UNKNOWN_REPOSITORY,
					"Unknown repository: " + repository, response);
		}
		catch (AccessDeniedException e) {
			_sendForbidden("Access denied", response);
		}
		catch (Exception e) {
			ThreadLog.error("Unknown error on removing RDF data from repository "
					+ repository, e);

			adminListener.error(
					"The server generated an unknown error; Please inform the server administrator",
					-1, -1, null);
		}
		finally {
			if (rdfRepository != null && rdfRepository.transactionStarted()) {
				rdfRepository.commitTransaction();
			}
			// FIXME is this always correct? can this still be reported if the response has 
			// already been committed to an error message (by one of the catch clauses).
			adminListener.transactionEnd();
		}
	}

	/*------------------------------*
	 * Inner class StatementRemover *
	 *------------------------------*/

	private static class StatementRemover implements StatementHandler {

		private RdfRepository _rdfRepository;

		public StatementRemover(RdfRepository rdfRepository) {
			_rdfRepository = rdfRepository;
		}

		// implements StatementHandler.handleStatement(...)
		public void handleStatement(Resource subject, URI predicate, Value object)
			throws StatementHandlerException
		{
			try {
				_rdfRepository.removeStatements(subject, predicate, object);
			}
			catch (SailUpdateException e) {
				throw new StatementHandlerException(e);
			}
		}

	}

	/*--------------------------*
	 * Inner class ErrorHandler *
	 *--------------------------*/

	private static class ErrorHandler implements org.openrdf.rio.ParseErrorListener {

		private int _errorCount;

		private int _statementCount;

		private AdminListener _adminListener;

		public ErrorHandler(AdminListener adminListener) {
			_errorCount = 0;
			_statementCount = 0;
			_adminListener = adminListener;
		}

		public int getErrorCount() {
			return _errorCount;
		}

		public int getStatementCount() {
			return _statementCount;
		}

		public void warning(String msg, int lineNo, int colNo) {
			_adminListener.notification(msg, lineNo, colNo, null);
		}

		public void error(String msg, int lineNo, int colNo) {
			_adminListener.warning(msg, lineNo, colNo, null);
			_errorCount++;
		}

		public void fatalError(String msg, int lineNo, int colNo) {
			_adminListener.error(msg, lineNo, colNo, null);
			_errorCount++;
		}
	}
}
