/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.rio.RdfDocumentWriter;
import org.openrdf.rio.n3.N3Writer;
import org.openrdf.rio.ntriples.NTriplesWriter;
import org.openrdf.rio.rdfxml.RdfXmlWriter;
import org.openrdf.rio.turtle.TurtleWriter;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.repository.local.LocalRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPErrorType;
import org.openrdf.sesame.server.SesameServer;

public class RdfExtractionServlet extends SesameServlet {

	protected void _doGet(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		_handleFormURLEncodedRequest(request, response);
	}

	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (x-www-form-urlencoded)
		String repository = HttpServerUtil.getParameter(request, "repository");
		String schema = HttpServerUtil.getParameter(request, "schema", "off");
		String data = HttpServerUtil.getParameter(request, "data", "off");
		String explicitOnly = HttpServerUtil.getParameter(request, "explicitOnly", "off");
		String niceOutput = HttpServerUtil.getParameter(request, "niceOutput", "off");
		String serialization = HttpServerUtil.getParameter(request, "serialization", RDFFormat.RDFXML.toString());

		_handleRequest(request, response, repository, schema, data, explicitOnly, niceOutput, serialization);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (multipart/form-data)
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		String repository = HttpServerUtil.getParameter(fileItemMap, "repository");
		String schema = HttpServerUtil.getParameter(fileItemMap, "schema", "off");
		String data = HttpServerUtil.getParameter(fileItemMap, "data", "off");
		String explicitOnly = HttpServerUtil.getParameter(fileItemMap, "explicitOnly", "off");
		String niceOutput = HttpServerUtil.getParameter(fileItemMap, "niceOutput", "off");
		String serialization = HttpServerUtil.getParameter(fileItemMap, "serialization", RDFFormat.RDFXML.toString());

		_handleRequest(request, response, repository, schema, data, explicitOnly, niceOutput, serialization);
	}

	private void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String repository, String schema, String data, String explicitOnly, String niceOutput, String serialization)
		throws IOException
	{
		SesameServer.setThreadLogFileForRepository(repository);
		_logIP(request);
		ThreadLog.log(">>> extract RDF");

		// form data trace output
		ThreadLog.trace("repository = " + repository);
		ThreadLog.trace("schema = " + schema);
		ThreadLog.trace("data = " + data);
		ThreadLog.trace("explicitOnly = " + explicitOnly);
		ThreadLog.trace("niceOutput = " + niceOutput);
		ThreadLog.trace("serialization = " + serialization);

		// Check request parameters
		if (repository == null) {
			_sendBadRequest("No repository specified", response);
			return;
		}

		boolean exportSchema = schema.equals("on");
		boolean exportData = data.equals("on");
		boolean explicit = explicitOnly.equals("on");
		boolean nice = niceOutput.equals("on");

		// Check the requested serialization and create an RdfDocumentWriter
		HTTPOutputStream httpOut = new HTTPOutputStream(response);
		httpOut.setCacheableResult(false);
		httpOut.setUseGZIPCompression(HttpServerUtil.acceptsGZIPEncoding(request));

		RdfDocumentWriter rdfDocWriter = null;
		RDFFormat rdfFormat = RDFFormat.forValue(serialization);

		if (rdfFormat == RDFFormat.RDFXML) {
			httpOut.setContentType("application/rdf+xml; charset=utf-8");
			Writer out = new OutputStreamWriter(httpOut, "UTF-8");
			rdfDocWriter = new RdfXmlWriter(out);
		}
		else if (rdfFormat == RDFFormat.NTRIPLES) {
			httpOut.setContentType("text/plain");
			Writer out = new OutputStreamWriter(httpOut, "UTF-8");
			rdfDocWriter = new NTriplesWriter(out);
		}
		else if (rdfFormat == RDFFormat.N3) {
			httpOut.setContentType("text/rdf+n3");
			Writer out = new OutputStreamWriter(httpOut, "UTF-8");
			rdfDocWriter = new N3Writer(out);
		}
		else if (rdfFormat == RDFFormat.TURTLE) {
			httpOut.setContentType("application/x-turtle");
			Writer out = new OutputStreamWriter(httpOut, "UTF-8");
			rdfDocWriter = new TurtleWriter(out);
		}
		else {
			_sendBadRequest("Unknown serialization: " + serialization, response);
			return;
		}

		try {
			LocalService service = SesameServer.getLocalService();

			_login(service);

			LocalRepository rep = (LocalRepository)service.getRepository(repository);

			// Extract the data
			long startTime = System.currentTimeMillis();

			rep.extractRDF(rdfDocWriter, exportSchema, exportData, explicit, nice);
			httpOut.close();

			long endTime = System.currentTimeMillis();
			ThreadLog.trace("RDF exported in " + (endTime - startTime) + "ms");
		}
		catch (UnknownRepositoryException e) {
			_sendBadRequest(HTTPErrorType.UNKNOWN_REPOSITORY, "Unknown repository: " + repository, response);
		}
		catch (AccessDeniedException e) {
			_sendForbidden("Access denied: " + e.getMessage(), response);
		}
		catch (IOException e) {
			ThreadLog.warning("I/O error, client connection closed?: " + e.getMessage());
		}
		catch (Exception e) {
			_sendInternalError("Unknown error on exporting RDF from repository " + repository, e, response);
		}
	}
}
