/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Map;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.omm.SessionContext;
import org.openrdf.sesame.server.SesameServer;

public class LoginServlet extends SesameServlet {

/*--------------------------------------------+
|	Constants                                 |
+--------------------------------------------*/

	/**
	 * The delay (in ms) for returning a failure message.
	 **/
	private static long FAILURE_DELAY = 3000;

	/**
	 * Key under which failure message will be stored in the session context.
	 **/
	private static String FAILURE_MSG_KEY = "loginFailedMsg";

/*--------------------------------------------+
|	Methods                                   |
+--------------------------------------------*/

	protected void _doGet(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		_handleFormURLEncodedRequest(request, response);
	}

	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (x-www-form-urlencoded)
		String user = HttpServerUtil.getParameter(request, "user");
		String password = HttpServerUtil.getParameter(request, "password");
		String successPage = HttpServerUtil.getParameter(request, "successPage");
		String failurePage = HttpServerUtil.getParameter(request, "failurePage");

		_handleRequest(request, response, user, password, successPage, failurePage);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (multipart/form-data)
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		String user = HttpServerUtil.getParameter(fileItemMap, "user");
		String password = HttpServerUtil.getParameter(fileItemMap, "password");
		String successPage = HttpServerUtil.getParameter(fileItemMap, "successPage");
		String failurePage = HttpServerUtil.getParameter(fileItemMap, "failurePage");

		_handleRequest(request, response, user, password, successPage, failurePage);
	}

	private void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String user, String password, String successPage, String failurePage)
		throws IOException
	{
		SesameServer.setThreadLogFile("login.log");
		_logIP(request);
		ThreadLog.log(">>> login");

		ThreadLog.trace("user = " + user);
		ThreadLog.trace("password set = " + (password != null));
		ThreadLog.trace("successPage = " + successPage);
		ThreadLog.trace("failurePage = " + failurePage);

		HttpServerUtil.setNoCacheHeaders(response);

		if (user == null) {
			String errMsg = "user parameter is missing";
			request.getSession().setAttribute(FAILURE_MSG_KEY, errMsg);
			_sendBadRequest(errMsg, response);
			return;
		}

		try {
			SesameServer.getLocalService().login(user, password);

			// No AccessDeniedException thrown, logged in successfully

			// Create a session ID
			HttpSession session = request.getSession();

			// Store session context under session ID
			SessionContext.put(session.getId(), SessionContext.getContext());

			// Store session ID in a cookie
			Cookie sessionCookie = new Cookie(SESSIONID, session.getId());
			sessionCookie.setPath(request.getContextPath() + "/");
			response.addCookie(sessionCookie);

			// Remove any failure messages that might have been set earlier:
			session.removeAttribute(FAILURE_MSG_KEY);

			ThreadLog.trace("User '" + user + "' logged in");

			if (successPage != null) {
				// send redirect
				response.sendRedirect(request.getContextPath() + successPage);
			}
			else {
				// send OK message
				response.setStatus(HttpServletResponse.SC_OK);
				response.setContentType("text/plain");
				PrintWriter writer = response.getWriter();
				writer.write("Logged in successfully");
				writer.close();
			}
		}
		catch (AccessDeniedException e) {
			String errMsg = e.getMessage();
			if (errMsg == null) {
				errMsg = "Access denied";
			}

			// failed to log in, set the error message
			request.getSession().setAttribute(FAILURE_MSG_KEY, errMsg);

			ThreadLog.trace("Login failed: " + errMsg);

			// Wait a little before returning the error message
			try {
				Thread.sleep(FAILURE_DELAY);
			} catch (InterruptedException ignore) {
				// ignore
			}

			if (failurePage != null) {
				// send redirect to failure page
				response.sendRedirect(request.getContextPath() + failurePage);
			}
			else {
				// send error message
				_sendForbidden(errMsg, response);
			}
		}
	}
}
