/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;
import java.io.OutputStream;

import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;

/**
 * An <tt>OutputStream</tt> that wraps an <tt>HttpServletResponse</tt>.
 * HTTPOutputStream is implemented such that it only set the status and
 * opens an output stream to a supplied <tt>HttpServletResponse</tt>
 * when the first byte is actually written.
 **/
public class HTTPOutputStream extends OutputStream {

/*--------------------+
| Variables           |
+--------------------*/

	private HttpServletResponse _response;
	private OutputStream _responseStream;
	private boolean _streamOpened;
	private boolean _streamClosed;
	private String _contentType;
	private boolean _cacheable;
	private boolean _useGZIP;

/*--------------------+
| Constructors        |
+--------------------*/

	/**
	 * Creates a new HTTPOutputStream.
	 **/
	public HTTPOutputStream(HttpServletResponse response) {
		this(response, null);
	}

	/**
	 * Creates a new HTTPOutputStream.
	 **/
	public HTTPOutputStream(HttpServletResponse response, String contentType) {
		_response = response;
		_contentType = contentType;
		_cacheable = true;
		_useGZIP = false;
		_streamOpened = false;
		_streamClosed = false;
	}

/*--------------------+
| Methods             |
+--------------------*/

	/**
	 * Sets the content type that will be reported to the client, e.g.
	 * <tt>text/html</tt>.
	 **/
	public void setContentType(String contentType) {
		_contentType = contentType;
	}
	
	/**
	 * Sets whether the receiver of the written result is allowed cache
	 * it. Setting this flag to <tt>false</tt> will result in a number
	 * of headers being added to the HTTP response.
	 **/
	public void setCacheableResult(boolean cacheable) {
		_cacheable = cacheable;
	}

	/**
	 * Sets whether written response should be compressed using gzip. This
	 * should only be enable if the sender of the request has indicated that
	 * it can handle gzip-encoded data.
	 **/
	public void setUseGZIPCompression(boolean useGZIP) {
		_useGZIP = useGZIP;
	}

	// implements OutputStream.write(int)
	public void write(int b)
		throws IOException
	{
		_ensureOpened();
		_responseStream.write(b);
	}

	// implements OutputStream.write(byte[])
	public void write(byte[] b)
		throws IOException
	{
		_ensureOpened();
		_responseStream.write(b);
	}

	// implements OutputStream.write(byte[], int, int)
	public void write(byte[] b, int off, int len)
		throws IOException
	{
		_ensureOpened();
		_responseStream.write(b, off, len);
	}

	// implements OutputStream.flush()
	public void flush()
		throws IOException
	{
		if (_responseStream != null) {
			_responseStream.flush();
			_response.flushBuffer();
		}
	}

	// implements OutputStream.close()
	public void close()
		throws IOException
	{
		// Close stream only once:
		if (!_streamClosed) {
			_ensureOpened();

			_responseStream.close();
			_response.flushBuffer();

			_streamClosed = true;
		}
	}

	/**
	 * Checks whether the response stream has been opened yet. Note
	 * that this method does not indicate that the stream is still
	 * open, it might has been closed again.
	 *
	 * @return <tt>true</tt> if the stream has been opened,
	 * <tt>false</tt> otherwise.
	 **/
	public boolean streamOpened() {
		return _streamOpened;
	}

	/**
	 * Checks whether the response stream has been closed.
	 *
	 * @return <tt>true</tt> if the stream has been closed,
	 * <tt>false</tt> otherwise.
	 **/
	public boolean streamClosed() {
		return _streamClosed;
	}

	/**
	 * Ensures that the output stream has been opened properly and
	 * that it is available via the variable _responseStream.
	 **/
	private void _ensureOpened()
		throws IOException
	{
		if (_streamClosed) {
			throw new IOException("OutputStream has already been closed");
		}

		if (!_streamOpened) {
			_response.setContentType(_contentType);
			if (!_cacheable) {
 				HttpServerUtil.setNoCacheHeaders(_response);
			}
			_response.setStatus(HttpServletResponse.SC_OK);

			if (_useGZIP) {
				_responseStream = HttpServerUtil.openGZIPOutputStream(_response);
			} else {
				_responseStream = _response.getOutputStream();
			}

			_streamOpened = true;
		}
	}
}
