/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.QueryLanguage;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.query.GraphQueryResultListener;
import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.RdfGraphWriter;
import org.openrdf.sesame.repository.SesameRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPErrorType;
import org.openrdf.sesame.server.SesameServer;

public class GraphQueryServlet extends SesameServlet {

	protected void _doGet(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		_handleFormURLEncodedRequest(request, response);
	}

	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (x-www-form-urlencoded)
		String repository = HttpServerUtil.getParameter(request, "repository");
		String query = HttpServerUtil.getParameter(request, "query");
		String queryLanguageStr = HttpServerUtil.getParameter(request, "queryLanguage");
		String serialization = HttpServerUtil.getParameter(request, "serialization", RDFFormat.RDFXML.toString());

		_handleRequest(request, response, repository, query, queryLanguageStr, serialization);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (multipart/form-data)
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		String repository = HttpServerUtil.getParameter(fileItemMap, "repository");
		String query = HttpServerUtil.getParameter(fileItemMap, "query");
		String queryLanguageStr = HttpServerUtil.getParameter(fileItemMap, "queryLanguage");
		String serialization = HttpServerUtil.getParameter(fileItemMap, "serialization", RDFFormat.RDFXML.toString());

		_handleRequest(request, response, repository, query, queryLanguageStr, serialization);
	}

	protected void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String repository, String query, String queryLanguageStr, String serialization)
		throws IOException
	{
		SesameServer.setThreadLogFileForRepository(repository);
		_logIP(request);
		ThreadLog.log(">>> evaluate graph query");

		// trace output
		ThreadLog.trace("repository = " + repository);
		ThreadLog.trace("query = " + query);
		ThreadLog.trace("queryLanguage = " + queryLanguageStr);
		ThreadLog.trace("serialization = " + serialization);

		// Check request parameters
		if (query == null) {
			_sendBadRequest("Query is missing", response);
			return;
		}

		if (repository == null) {
			_sendBadRequest("No repository specified", response);
			return;
		}

		if (queryLanguageStr == null) {
			_sendBadRequest("No query language specified", response);
			return;
		}

		QueryLanguage queryLanguage = QueryLanguage.forValue(queryLanguageStr);
		if (queryLanguage == null) {
			_sendBadRequest("Unsupported query language: " + queryLanguageStr, response);
			return;
		}

		HTTPOutputStream httpOut = new HTTPOutputStream(response);
		httpOut.setUseGZIPCompression(HttpServerUtil.acceptsGZIPEncoding(request));
		httpOut.setCacheableResult(false);

		RDFFormat rdfFormat = RDFFormat.forValue(serialization);
		if (rdfFormat == RDFFormat.RDFXML) {
			httpOut.setContentType("application/rdf+xml");
		}
		else if (rdfFormat == RDFFormat.NTRIPLES) {
			httpOut.setContentType("text/plain");
		}
		else if (rdfFormat == RDFFormat.N3) {
			httpOut.setContentType("text/rdf+n3");
		}
		else if (rdfFormat == RDFFormat.TURTLE) {
			httpOut.setContentType("application/x-turtle");
		}
		else {
			_sendBadRequest("Unknown serialization format: " + serialization, response);
			return;
		}

		GraphQueryResultListener graphListener = new RdfGraphWriter(rdfFormat, httpOut);

		try {
			LocalService service = SesameServer.getLocalService();

			_login(service);

			SesameRepository rep = service.getRepository(repository);

			// Evaluate the query
			rep.performGraphQuery(queryLanguage, query, graphListener);
		}
		catch (UnknownRepositoryException e) {
			_sendBadRequest(HTTPErrorType.UNKNOWN_REPOSITORY, "Unknown repository: " + repository, response);
		}
		catch (AccessDeniedException e) {
			_sendForbidden("Access denied: " + e.getMessage(), response);
		}
		catch (MalformedQueryException e) {
			_sendBadRequest(HTTPErrorType.MALFORMED_QUERY, e.getMessage(), response);
		}
		catch (QueryEvaluationException e) {
			ThreadLog.warning("Query evaluation error: " + e.getMessage());
			if (!httpOut.streamOpened()) {
				_sendBadRequest(HTTPErrorType.QUERY_EVALUATION_ERROR, e.getMessage(), response);
			}
			else {
				// Data has already been sent to the client, we can't send an
				// HTTP error code anymore.
				graphListener.reportError("Query evaluation error: " + e.getMessage());
				graphListener.endGraphQueryResult();
			}
		}
		catch (IOException ie) {
			ThreadLog.warning("I/O error, client connection closed?\n" + ie.getMessage());
		}
		catch (Exception e) {
			ThreadLog.error("Unknown error on query: " + query, e);

			graphListener.reportError("The server generated an unknown error. Please inform the server administrator.");
			graphListener.endGraphQueryResult();
		}
	}
}
