/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.admin.AdminListener;
import org.openrdf.sesame.admin.HtmlAdminMsgWriter;
import org.openrdf.sesame.admin.XmlAdminMsgWriter;
import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.ConfigurationException;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.AdminResultFormat;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.repository.SesameRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPErrorType;
import org.openrdf.sesame.server.SesameServer;

public class DataUploadServlet extends SesameServlet {
	
/*------------------------------------------+
|	Methods                                 |
+------------------------------------------*/

	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (x-www-form-urlencoded)
		String repository = HttpServerUtil.getParameter(request, "repository");
		String data = HttpServerUtil.getParameter(request, "data");
		String baseURI = HttpServerUtil.getParameter(request, "baseURI");
		String dataFormatStr = HttpServerUtil.getParameter(request, "dataFormat", RDFFormat.RDFXML.toString());
		String resultFormatStr = HttpServerUtil.getParameter(request, "resultFormat", AdminResultFormat.XML.toString());
		String verify = HttpServerUtil.getParameter(request, "verifyData", "off");

		InputStream rdfStream = new ByteArrayInputStream(data.getBytes());
		_handleRequest(request, response, repository, rdfStream, baseURI, dataFormatStr, resultFormatStr, verify);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (multipart/form-data)
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		String repository = HttpServerUtil.getParameter(fileItemMap, "repository");
		InputStream rdfStream = HttpServerUtil.getStreamParameter(fileItemMap, "data");
		String baseURI = HttpServerUtil.getParameter(fileItemMap, "baseURI");
		String dataFormatStr = HttpServerUtil.getParameter(fileItemMap, "dataFormat", RDFFormat.RDFXML.toString());
		String resultFormatStr = HttpServerUtil.getParameter(fileItemMap, "resultFormat", AdminResultFormat.XML.toString());
		String verify = HttpServerUtil.getParameter(fileItemMap, "verifyData", "off");

		_handleRequest(request, response, repository, rdfStream, baseURI, dataFormatStr, resultFormatStr, verify);
	}

	private void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String repository, InputStream rdfStream, String baseURI, String dataFormatStr, String resultFormatStr, String verify)
		throws IOException
	{
		SesameServer.setThreadLogFileForRepository(repository);
		_logIP(request);
		ThreadLog.log(">>> upload data");

		// Log parameters
		ThreadLog.trace("repository = " + repository);
		ThreadLog.trace("baseURI = " + baseURI);
		ThreadLog.trace("dataFormat = " + dataFormatStr);
		ThreadLog.trace("resultFormat = " + resultFormatStr);
		ThreadLog.trace("verifyData = " + verify);

		// Check parameters
		if (rdfStream == null) {
			_sendBadRequest("RDF data is missing", response);
			return;
		}

		if (baseURI == null) {
			// Default to foo:bar
			baseURI = "foo:bar";
		}

		if (repository == null) {
			_sendBadRequest("No repository specified", response);
			return;
		}
		
		// Check the data format
		RDFFormat dataFormat = RDFFormat.forValue(dataFormatStr);
		if (dataFormat != RDFFormat.RDFXML &&
				dataFormat != RDFFormat.NTRIPLES &&
				dataFormat != RDFFormat.TURTLE)
		{
			_sendBadRequest("Unsupported data format: " + dataFormatStr, response);
			return;
		}
		
		// Check the reponse format and create an AdminListener
		HTTPOutputStream httpOut = new HTTPOutputStream(response);
		httpOut.setCacheableResult(false);
		
		AdminListener report = null;
		AdminResultFormat resultFormat = AdminResultFormat.forValue(resultFormatStr);
		
		if (resultFormat == AdminResultFormat.XML) {
			httpOut.setContentType("text/xml");
			report = new XmlAdminMsgWriter(httpOut);
		}
		else if (resultFormat == AdminResultFormat.HTML) {
			httpOut.setContentType("text/html");
			report = new HtmlAdminMsgWriter(httpOut);
		}
		else {
			_sendBadRequest("Unknown result format: " + resultFormatStr, response);
			return;
		}
		
		// Check whether the input should be verified
		boolean verifyData = verify.equals("on");
		
		try {
			LocalService service = SesameServer.getLocalService();
			
			_login(service);
			
			SesameRepository rep = service.getRepository(repository);
			
			// Add the data
			long startTime = System.currentTimeMillis();
			
			rep.addData(rdfStream, baseURI, dataFormat, verifyData, report);
			rdfStream.close();
			
			long endTime = System.currentTimeMillis();
			ThreadLog.trace("data added in " + (endTime - startTime) + "ms");
		}
		catch (AccessDeniedException e) {
			_sendForbidden("Access denied", response);
		}
		catch (ConfigurationException e) {
			_sendInternalError("Configuration error", e, response);
		}
		catch (UnknownRepositoryException e) {
			_sendBadRequest(HTTPErrorType.UNKNOWN_REPOSITORY, "Unknown repository: " + repository, response);
		}
		catch (RuntimeException e) {
			ThreadLog.error("Unknown error on adding RDF data to repository " + repository, e);
		}
	} 
}
