/*
 * Sesame - Storage and Querying architecture for RDF and RDF Schema 
 * Copyright (C) 2001-2006 Aduna
 * 
 * Contact: 
 * 	Aduna 
 *  Prinses Julianaplein 14 b 
 *  3817 CS Amersfoort 
 *  The Netherlands
 *  tel. +33 (0)33 465 99 87 
 *  fax. +33 (0)33 465 99 87
 * 
 * http://aduna-software.com/ 
 * http://www.openrdf.org/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package org.openrdf.sesame.server.http;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.rio.ParseErrorListener;
import org.openrdf.rio.Parser;
import org.openrdf.rio.StatementHandler;
import org.openrdf.rio.StatementHandlerException;
import org.openrdf.rio.ntriples.NTriplesParser;
import org.openrdf.rio.rdfxml.RdfXmlParser;
import org.openrdf.rio.turtle.TurtleParser;

import org.openrdf.sesame.admin.AdminListener;
import org.openrdf.sesame.admin.HtmlAdminMsgWriter;
import org.openrdf.sesame.admin.XmlAdminMsgWriter;
import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.AdminResultFormat;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.repository.local.LocalRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPErrorType;
import org.openrdf.sesame.sail.RdfRepository;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.server.SesameServer;

public class DataMergeServlet extends SesameServlet {

	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters
		String repository = HttpServerUtil.getParameter(request, "repository");
		String resultFormatStr = HttpServerUtil.getParameter(request, "resultFormat", AdminResultFormat.XML.toString());

		String data = HttpServerUtil.getParameter(request, "data");
		String baseURI = HttpServerUtil.getParameter(request, "baseURI", "foo:bar");
		String dataFormatStr = HttpServerUtil.getParameter(request, "dataFormat");

		_handleRequest(request, response, repository, resultFormatStr, 
						data, baseURI, dataFormatStr);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		// Get request parameters (multipart/form-data)
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		// Get request parameters
		String repository = HttpServerUtil.getParameter(fileItemMap, "repository");
		String resultFormatStr = HttpServerUtil.getParameter(fileItemMap, "resultFormat", AdminResultFormat.XML.toString());

		String data = HttpServerUtil.getParameter(fileItemMap, "data");
		String baseURI = HttpServerUtil.getParameter(fileItemMap, "baseURI", "foo:bar");
		String dataFormatStr = HttpServerUtil.getParameter(fileItemMap, "dataFormat");

		_handleRequest(request, response, repository, resultFormatStr, data, baseURI, dataFormatStr);
	}

	private void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String repository, String resultFormatStr, 
		String data, String baseURI, String dataFormatStr)
		throws IOException
	{
		SesameServer.setThreadLogFileForRepository(repository);
		_logIP(request);
		ThreadLog.log(">>> merge data");

		// Log parameters
		ThreadLog.trace("repository = " + repository);
		ThreadLog.trace("resultFormat = " + resultFormatStr);
		ThreadLog.trace("data file sent: " + (data != null));
		ThreadLog.trace("dataFormat = " + dataFormatStr);

		// Check repository parameter
		if (repository == null) {
			_sendBadRequest("No repository specified", response);
			return;
		}
		
		if (data == null) {
			_sendBadRequest("No data specified", response);
			return;
		}

		// Check the result format and create an AdminListener
		HTTPOutputStream httpOut = new HTTPOutputStream(response);
		httpOut.setCacheableResult(false);

		AdminResultFormat resultFormat = AdminResultFormat.forValue(resultFormatStr);
		AdminListener report = null;

		if (resultFormat == AdminResultFormat.XML) {
			httpOut.setContentType("text/xml");
			report = new XmlAdminMsgWriter(httpOut);
		}
		else if (resultFormat == AdminResultFormat.HTML) {
			httpOut.setContentType("text/html");
			report = new HtmlAdminMsgWriter(httpOut);
		}
		else {
			_sendBadRequest("Unknown result format: " + resultFormatStr, response);
			return;
		}

		RdfRepository rdfRepository = null;

		try {
			LocalService service = SesameServer.getLocalService();
			
			_login(service);
			
			LocalRepository rep = (LocalRepository)service.getRepository(repository);
			
			// Explicit check for write access is necessary, since from
			// this point on we bypass the security mechanism and work directly on the
			// SAIL.
			if (! rep.hasWriteAccess()) {
				_sendForbidden("Access denied", response);
				return;
			}
			
			rdfRepository = (RdfRepository)rep.getSail();
			
			// check data format
			if (dataFormatStr == null) {
				_sendBadRequest("no data serialization format specified", response);
				return;
			}
			
			Parser parser = null;
			RDFFormat dataFormat = RDFFormat.forValue(dataFormatStr);
			
			if (dataFormat.equals(RDFFormat.TURTLE)) {
				parser = new TurtleParser();
			}
			else if (dataFormat.equals(RDFFormat.RDFXML)) {
				parser = new RdfXmlParser();
			}
			else if (dataFormat.equals(RDFFormat.NTRIPLES)) {
				parser = new NTriplesParser();
			}
			else {
				_sendBadRequest("unknown data serialization format: " + dataFormatStr, response);
				return;
			}
			
			StatementHandler merger = new StatementMerger(rdfRepository);
			parser.setStatementHandler(merger);
			ParseErrorListener errorListener = new ErrorHandler(report);
			parser.setParseErrorListener(errorListener);
			parser.setPreserveBNodeIds(true);
			
			Reader reader = new StringReader(data);
			
			rdfRepository.startTransaction();
			report.transactionStart();
			parser.parse(reader, baseURI);
			
		}
		catch (UnknownRepositoryException e) {
			_sendBadRequest(HTTPErrorType.UNKNOWN_REPOSITORY,
					"Unknown repository: " + repository, response);
		}
		catch (AccessDeniedException e) {
			_sendForbidden("Access denied", response);
		}
		catch (Exception e) {
			ThreadLog.error("Unknown error on removing RDF data from repository "
					+ repository, e);

			report.error(
					"The server generated an unknown error; Please inform the server administrator",
					-1, -1, null);
		}
		finally {
			if (rdfRepository != null && rdfRepository.transactionStarted()) {
				rdfRepository.commitTransaction();
			}
			// FIXME is this always correct? can this still be reported if the response has 
			// already been committed to an error message (by one of the catch clauses).
			report.transactionEnd();
		}
	}

	private static class StatementMerger implements StatementHandler {

		RdfRepository _rep;

		public StatementMerger(RdfRepository rep) {
			_rep = rep;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.openrdf.rio.StatementHandler#handleStatement(org.openrdf.model.Resource,
		 *      org.openrdf.model.URI, org.openrdf.model.Value)
		 */
		public void handleStatement(Resource subject, URI predicate, Value object)
			throws StatementHandlerException
		{
			try {
				_rep.addStatement(subject, predicate, object);
			}
			catch (SailUpdateException e) {
				throw new StatementHandlerException(e);
			}
		}

	}

	private static class ErrorHandler implements org.openrdf.rio.ParseErrorListener {

		private int _errorCount;

		private int _statementCount;

		private AdminListener _report;

		public ErrorHandler(AdminListener report) {
			_errorCount = 0;
			_statementCount = 0;
			_report = report;
		}

		public int getErrorCount() {
			return _errorCount;
		}

		public void resetErrorCount() {
			_errorCount = 0;
		}

		public int getStatementCount() {
			return _statementCount;
		}

		public void resetStatementCount() {
			_statementCount = 0;
		}

		public void warning(String msg, int lineNo, int colNo) {
			_report.notification(msg, lineNo, colNo, null);
		}

		public void error(String msg, int lineNo, int colNo) {
			_report.warning(msg, lineNo, colNo, null);
			_errorCount++;
		}

		public void fatalError(String msg, int lineNo, int colNo) {
			_report.error(msg, lineNo, colNo, null);
			_errorCount++;
		}
	}
}
